# hobo - portal to configure and deploy applications
# Copyright (C) 2015-2019  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django import forms
from django.conf import settings
from django.contrib import messages
from django.http import HttpResponseRedirect
from django.template.defaultfilters import slugify
from django.utils.translation import gettext_lazy as _

from .models import Authentic, BiJoe, Chrono, Combo, Fargo, Hobo, Lingo, Passerelle, Variable, Wcs, Welco
from .utils import get_variable

EXCLUDED_FIELDS = (
    'last_operational_check_timestamp',
    'last_operational_success_timestamp',
    'legacy_urls',
    'secret_key',
    'secondary',
)


class SelectDisabledChoices(forms.Select):
    # Select widget, able to display disabled choices
    # Choices may be (value, label) or (value, label, disabled) with disabled
    # a boolean set to True if the choice is disabled.

    def __init__(self, *args, choices=None, **kwargs):
        if choices:
            self.disabled_choices = [choice[0] for choice in choices if len(choice) > 2 and choice[2]]
            choices = [choice[:2] for choice in choices]
        super().__init__(*args, choices=choices, **kwargs)

    def create_option(self, *args, **kwargs):
        option = super().create_option(*args, **kwargs)
        if option.get('value') in self.disabled_choices:
            option['attrs']['disabled'] = 'disabled'
        return option


class BaseForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        choices = self.get_template_choices()
        if len(choices) < 2:
            del self.fields['template_name']
        else:
            self.fields['template_name'].choices = choices
            self.fields['template_name'].widget = SelectDisabledChoices(
                choices=choices,
            )
        if self.instance.id:
            del self.fields['slug']
            del self.fields['base_url']

    def get_template_choices(self):
        if not settings.SERVICE_TEMPLATES:
            return []
        service_id = self.Meta.model.Extra.service_id
        return settings.SERVICE_TEMPLATES.get(service_id, [])

    def save(self, commit=True):
        if not self.instance.slug:
            base_slug = slugify(self.instance.title)
            slug = base_slug
            i = 1
            while True:
                try:
                    self.Meta.model.objects.get(slug=slug)
                except self.Meta.model.DoesNotExist:
                    break
                i += 1
                slug = '%s-%s' % (base_slug, i)
            self.instance.slug = slug
        choices = self.get_template_choices()
        if not self.instance.id and len(choices) == 1:
            self.instance.template_name = choices[0][0]
        return super().save(commit=commit)


class AuthenticForm(BaseForm):
    class Meta:
        model = Authentic
        exclude = EXCLUDED_FIELDS

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if not self.instance.is_operational() and not self.initial.get('use_as_idp_for_self'):
            del self.fields['use_as_idp_for_self']

    def save(self, commit=True):
        if not self.instance.use_as_idp_for_self:
            if self.cleaned_data.get('use_as_idp_for_self'):
                # this idp was just marked as the idp to use, unmark all others
                Authentic.objects.update(use_as_idp_for_self=False)
        return super().save(commit=commit)


class WcsForm(BaseForm):
    class Meta:
        model = Wcs
        exclude = EXCLUDED_FIELDS


class PasserelleForm(BaseForm):
    class Meta:
        model = Passerelle
        exclude = EXCLUDED_FIELDS


class ComboForm(BaseForm):
    class Meta:
        model = Combo
        exclude = EXCLUDED_FIELDS

    def get_choice(self, choice):
        name, label = choice
        existing = self.Meta.model.objects.all().values_list('template_name', flat=True)
        forbidden = [
            template_name for template_name in existing if template_name in ('portal-user', 'portal-agent')
        ]
        if name == self.instance.template_name or name not in forbidden:
            return (name, label, False)
        return (name, label + _(' (allready used by another portal)'), True)

    def get_template_choices(self):
        return [self.get_choice(choice) for choice in super().get_template_choices()]

    def clean_template_name(self):
        template_name = self.cleaned_data['template_name']
        if template_name not in [
            name for name, dummy, disabled in self.get_template_choices() if not disabled
        ]:
            raise forms.ValidationError(_('This template cannot be used'))
        return template_name


class FargoForm(BaseForm):
    class Meta:
        model = Fargo
        exclude = EXCLUDED_FIELDS


class WelcoForm(BaseForm):
    class Meta:
        model = Welco
        exclude = EXCLUDED_FIELDS


class ChronoForm(BaseForm):
    class Meta:
        model = Chrono
        exclude = EXCLUDED_FIELDS


class LingoForm(BaseForm):
    class Meta:
        model = Lingo
        exclude = EXCLUDED_FIELDS


class BiJoeForm(BaseForm):
    class Meta:
        model = BiJoe
        exclude = EXCLUDED_FIELDS


class HoboForm(BaseForm):
    class Meta:
        model = Hobo
        exclude = EXCLUDED_FIELDS


class VariableForm(forms.ModelForm):
    class Meta:
        model = Variable
        exclude = ('service_type', 'service_pk', 'auto')

    def __init__(self, service=None, **kwargs):
        self.service = service
        super().__init__(**kwargs)

    def clean_name(self):
        name = self.cleaned_data['name']
        if not Variable.is_valid_name(name):
            raise forms.ValidationError(_('This name is not allowed for security reason'))
        return name

    def save(self, commit=True):
        if self.service:
            self.instance.service = self.service
        return super().save(commit=commit)


class VariablesFormMixin:
    form_class = None
    success_message = None
    variables = []

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        if self.request.POST:
            form_data = self.request.POST
        else:
            form_data = None
        initial_data = {}
        for variable_name in self.variables:
            initial_data[variable_name] = get_variable(variable_name).value
        context['form'] = self.form_class(form_data, initial=initial_data)
        return context

    def post(self, request, *args, **kwargs):
        form = self.form_class(request.POST)
        if not form.is_valid():
            return self.get(request, *args, **kwargs)
        changed = False
        for variable_name in self.variables:
            variable = get_variable(variable_name)
            if variable.value != form.cleaned_data[variable_name]:
                variable.value = form.cleaned_data[variable_name]
                variable.save()
                changed = True
        if changed and self.success_message:
            messages.info(self.request, self.success_message)

        return HttpResponseRedirect('.')


class ImportForm(forms.Form):
    parameters_json = forms.FileField(label=_('Parameters Export File'))
