# hobo - portal to configure and deploy applications

from unittest.mock import Mock

from django.db import connection


def test_json(app, settings):
    app.get('/.well-known/assetlinks.json', status=404)
    settings.WELL_KNOWN = {
        'assetlinks.json': {
            'content': [
                {
                    'a': 'b',
                }
            ]
        }
    }
    resp = app.get('/.well-known/assetlinks.json')
    assert resp.json == [
        {
            'a': 'b',
        }
    ]
    assert resp['content-type'] == 'application/json'
    settings.WELL_KNOWN['assetlinks.json']['content_type'] = 'application/json+entrouvert'
    resp = app.get('/.well-known/assetlinks.json')
    assert resp['content-type'] == 'application/json+entrouvert'

    settings.WELL_KNOWN['assetlinks.json']['content'] = set()
    resp = app.get('/.well-known/assetlinks.json')
    assert resp.json == {'err': 'JSON serialization failed'}


def test_text(app, settings):
    app.get('/.well-known/foobar', status=404)
    settings.WELL_KNOWN = {
        'foobar': {
            'content': 'coucou',
        }
    }
    resp = app.get('/.well-known/foobar')
    assert resp.text == 'coucou'
    assert resp['content-type'] == 'text/plain'

    settings.WELL_KNOWN['foobar']['content_type'] = 'text/xml'
    resp = app.get('/.well-known/foobar')
    assert resp['content-type'] == 'text/xml'


def test_directory(app, settings, tmp_path, monkeypatch):
    well_known_dir = tmp_path / 'well-known'
    well_known_dir.mkdir()

    foobar = well_known_dir / 'foo-bar'
    foobar.write_text('foobar')

    foo_dir = well_known_dir / 'foo'
    foo_dir.mkdir()

    bar_txt = foo_dir / 'bar.txt'
    bar_txt.write_text('barfoo')

    bar_json = foo_dir / 'bar.json'
    bar_json.write_text('{"foo": "bar"}')

    tenant = Mock()
    tenant.get_directory.return_value = str(tmp_path)

    monkeypatch.setattr(connection, 'tenant', tenant, raising=False)

    resp = app.get('/.well-known/foo-bar')
    assert resp.text == 'foobar'
    assert resp['content-type'] == 'application/octet-stream'

    resp = app.get('/.well-known/foo/bar.txt')
    assert resp.text == 'barfoo'
    assert resp['content-type'] == 'text/plain'

    resp = app.get('/.well-known/foo/bar.json')
    assert resp.json == {'foo': 'bar'}
    assert resp['content-type'] == 'application/json'

    app.get('/.well-known/does-not-exist', status=404)
