# hobo - portal to configure and deploy applications
# Copyright (C) 2015-2016 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import threading
import time
from importlib import import_module

from django.conf import UserSettingsHolder
from django.db import connection

from hobo.middleware.utils import StoreRequestMiddleware


def import_class(klass):
    module, attr = klass.rsplit('.', 1)
    return getattr(import_module(module), attr)


class TenantSettingsWrapper:
    local = threading.local()

    def __init__(self, default_settings):
        self.__dict__['tenants_settings'] = {}
        self.__dict__['default_settings'] = default_settings

    @property
    def loaders(self):
        loaders = getattr(self.default_settings, 'TENANT_SETTINGS_LOADERS', [])
        for loader in loaders:
            loader_class = import_class(loader)
            yield loader_class()

    def load_tenant_settings(self, tenant):
        """Load tenant settings from loaders into tenant_settings object"""
        tenant_settings = UserSettingsHolder(self.default_settings)
        for loader in self.loaders:
            loader.update_settings(tenant_settings, tenant)
        return tenant_settings

    def get_digest(self, tenant):
        return hash(tuple(loader.get_new_time(tenant) for loader in self.loaders))

    def get_tenant_settings(self, tenant):
        """Get last loaded settings for tenant.
        * if younger than 3 seconds use it,
        * otherwise compute digest from get_new_time() of each loader,
        * if digest has changed reload,
        * otherwise use last loaded setting and update last_time.
        """
        update_time = time.time()
        tenant_settings, last_digest, last_time = self.tenants_settings.get(
            tenant.domain_url, (None, None, None)
        )
        if last_digest and update_time - last_time < 3:
            return tenant_settings

        new_digest = self.get_digest(tenant)
        if new_digest != last_digest:
            tenant_settings = self.load_tenant_settings(tenant)

        self.tenants_settings[tenant.domain_url] = tenant_settings, new_digest, update_time
        return tenant_settings

    def get_tenant(self):
        # micro-optimization, connection use asgiref.local.Local as a thread
        # local storage, which is slow, use
        # StoreRequestMiddleware.get_request() to get the tenant from the
        # request to be a little faster
        request = StoreRequestMiddleware.get_request()
        tenant = getattr(request, 'tenant', None)
        if tenant is not None:
            return tenant
        # in other contexts without a request, use connection.tenant
        return connection.tenant

    def get_wrapped(self):
        if getattr(self.local, 'in_get_wrapped', False):
            return self.default_settings
        try:
            self.local.in_get_wrapped = True
            tenant = self.get_tenant()
            if not hasattr(tenant, 'domain_url'):
                return self.default_settings
            return self.get_tenant_settings(tenant)
        finally:
            self.local.in_get_wrapped = False

    def __getattr__(self, name):
        return getattr(self.get_wrapped(), name)

    def __setattr__(self, name, value):
        setattr(self.get_wrapped(), name, value)

    def __delattr__(self, name):
        delattr(self.get_wrapped(), name)
