# hobo - portal to configure and deploy applications
# Copyright (C) 2019  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.contrib import messages
from django.urls import reverse_lazy
from django.utils.translation import gettext_lazy as _
from django.views.generic import FormView

from .forms import EnableForm, SettingsForm
from .utils import (
    MatomoError,
    MatomoException,
    MatomoWS,
    auto_configure_matomo,
    compute_cnil_acknowledgment_level,
    get_tracking_js,
    get_variable,
    get_variable_value,
    put_tracking_js,
)


class EnableManualView(FormView):
    form_class = SettingsForm
    template_name = 'hobo/matomo_enable_manual.html'
    success_url = reverse_lazy('matomo-home')

    def get_initial(self):
        initial = super().get_initial()
        for suffix in ('tracking_js', 'tracking_js_backoffice'):
            initial[suffix] = get_tracking_js(suffix=suffix)
        return initial

    def form_valid(self, form):
        for suffix in ('tracking_js', 'tracking_js_backoffice'):
            tracking_js = form.cleaned_data[suffix]
            put_tracking_js(tracking_js, suffix=suffix)
        logme_url = get_variable('matomo_logme_url')
        logme_url.delete()
        return super().form_valid(form)


enable_manual = EnableManualView.as_view()


class HomeView(EnableManualView):
    template_name = 'hobo/matomo_home.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['enabled'] = False
        logme_url = get_variable_value('matomo_logme_url')
        context['logme_url'] = logme_url
        context['cnil_ack_levels'] = cnil_ack_levels = []
        for suffix in ('tracking_js', 'tracking_js_backoffice'):
            tracking_js = get_tracking_js(suffix=suffix)
            # compute contextual values
            space = _('Backoffice') if 'backoffice' in suffix else _('User portal')
            cnil_ack_level = compute_cnil_acknowledgment_level(tracking_js)
            if cnil_ack_level:
                cnil_ack_levels.append((space, cnil_ack_level))
            if not context['enabled']:
                context['enabled'] = bool(tracking_js)
        try:
            MatomoWS()
        except MatomoError:
            context['ws_available'] = False
        else:
            context['ws_available'] = True
        if logme_url:
            context['mode'] = 'auto'
        else:
            context['mode'] = 'manual'
        return context


home = HomeView.as_view()


class EnableAutoView(FormView):
    form_class = EnableForm
    template_name = 'hobo/matomo_enable_auto.html'
    success_url = reverse_lazy('matomo-home')

    def form_valid(self, form):
        matomo = MatomoWS()
        try:
            results = auto_configure_matomo(matomo, templates=('portal-user', 'portal-agent'))
        except MatomoException as exc:
            messages.error(self.request, 'matomo: ' + str(exc))
        else:
            for template, idsite in results:
                try:
                    matomo.create_fake_first_tracking_visit(idsite)
                except MatomoException as exc:
                    messages.warning(self.request, f'ping for {template}: {exc!s}')
        return super().form_valid(form)


enable_auto = EnableAutoView.as_view()


class DisableView(FormView):
    form_class = EnableForm
    template_name = 'hobo/matomo_disable.html'
    success_url = reverse_lazy('matomo-home')

    def form_valid(self, form):
        for suffix in ('tracking_js', 'tracking_js_backoffice'):
            put_tracking_js('', suffix=suffix)
        variable = get_variable('matomo_logme_url')
        variable.delete()
        return super().form_valid(form)


disable = DisableView.as_view()
