# hobo - portal to configure and deploy applications
# Copyright (C) 2015-2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import re

from rest_framework import permissions
from rest_framework.response import Response
from rest_framework.views import APIView

from hobo.rest_permissions import IsAdminUser, IsAPIClient

from .models import Variable

APIAdminOrAuthUser = (IsAPIClient & IsAdminUser) | ((~IsAPIClient) & permissions.IsAuthenticated)


class VariablesAPI(APIView):
    permission_classes = (APIAdminOrAuthUser,)

    def post(self, request, format=None):
        errors = []
        for key, value in request.data.items():
            if key.startswith('SETTING_') and key not in Variable.ALLOWED_SETTING_VALUES:
                errors.append('"%s" is not allowed.' % key)
            elif not re.match(r'^[a-zA-Z][\w\d]+$', key):
                errors.append('"%s" is not a valid variable key.' % key)

        if errors:
            return Response({'err': 1, 'errors': errors}, status=400)

        for key, value in request.data.items():
            variable, dummy = Variable.objects.get_or_create(name=key, defaults={'auto': True})
            if isinstance(value.get('value'), str):
                variable.value = value.get('value')
            else:
                variable.json = value.get('value')
            if value.get('label'):
                variable.label = value.get('label')
            variable.save()
        return Response({'err': 0})


variables = VariablesAPI.as_view()
