import uuid

from django.contrib.auth import get_user_model
from django.contrib.auth.models import Group
from django.contrib.contenttypes.models import ContentType
from django.db import connection
from django.db.migrations.executor import MigrationExecutor

from combo.data.models import Page, TextCell, UniverCell


def test_generic_migration(transactional_db):
    page = Page.objects.create(title='One', slug='index')
    group1 = Group.objects.create(name='plop1')

    migrate_from = [('data', '0075_univercell')]
    migrate_to = [('data', '0077_convert_textcell_to_univercell')]

    executor = MigrationExecutor(connection)
    old_apps = executor.loader.project_state(migrate_from).apps
    executor.migrate(migrate_from)

    OldFeedCell = old_apps.get_model('data', 'FeedCell')
    old_feed_cell = OldFeedCell.objects.create(
        uuid=str(uuid.uuid4()),
        page_id=page.id,
        order=1,
        title='a title',
        url='http://example.net',
    )
    old_feed_cell.groups.set([group1.id])  # will force an extra save()

    old_other_feed_cell = OldFeedCell.objects.create(
        uuid=str(uuid.uuid4()),
        page_id=page.id,
        order=2,
        title='another title',
        url='http://example.net',
    )

    OldTextCell = old_apps.get_model('data', 'TextCell')
    old_text_cell = OldTextCell.objects.create(
        uuid=str(uuid.uuid4()),
        page_id=page.id,
        order=1,
        title='a title',
        text='hello',
    )

    executor = MigrationExecutor(connection)
    executor.migrate(migrate_to)
    executor.loader.build_graph()

    cells = UniverCell.objects.all().order_by('id')
    assert len(cells) == 3
    assert cells[0].attributes == {'title': 'a title', 'url': 'http://example.net', 'limit': None}
    assert cells[0].class_name == 'combo.data.models.FeedCell'
    assert str(cells[0].uuid) == str(old_feed_cell.uuid)
    assert [x.id for x in cells[0].groups.all()] == [group1.id]

    assert cells[1].attributes == {'title': 'another title', 'url': 'http://example.net', 'limit': None}
    assert cells[1].class_name == 'combo.data.models.FeedCell'
    assert str(cells[1].uuid) == str(old_other_feed_cell.uuid)
    assert [x.id for x in cells[1].groups.all()] == []

    assert cells[2].attributes == {'title': 'a title', 'text': 'hello'}
    assert cells[2].class_name == 'combo.data.models.TextCell'
    assert str(cells[2].uuid) == str(old_text_cell.uuid)

    # test backward
    executor = MigrationExecutor(connection)
    executor.migrate(migrate_from)
    assert UniverCell.objects.count() == 0


def test_kb_migration(transactional_db):
    page = Page.objects.create(title='One', slug='index')

    migrate_from = [('kb', '0008_cell_uuid')]
    migrate_to = [('kb', '0009_convert_to_univercell')]

    executor = MigrationExecutor(connection)
    old_apps = executor.loader.project_state(migrate_from).apps
    executor.migrate(migrate_from)

    OldLatestPageUpdatesCell = old_apps.get_model('kb', 'LatestPageUpdatesCell')
    old_cell = OldLatestPageUpdatesCell.objects.create(
        uuid=str(uuid.uuid4()),
        page_id=page.id,
        root_page_id=page.id,
        order=1,
        limit=5,
    )

    executor = MigrationExecutor(connection)
    executor.migrate(migrate_to)
    executor.loader.build_graph()

    cells = UniverCell.objects.all()
    assert len(cells) == 1
    assert cells[0].attributes == {'limit': 5, 'root_page': page.id}
    assert cells[0].class_name == 'combo.apps.kb.models.LatestPageUpdatesCell'
    assert str(cells[0].uuid) == str(old_cell.uuid)


def test_menucell_migration(transactional_db):
    page = Page.objects.create(title='One', slug='index')

    migrate_from = [('data', '0075_univercell')]
    migrate_to = [('data', '0081_convert_menucell_to_univercell')]

    executor = MigrationExecutor(connection)
    old_apps = executor.loader.project_state(migrate_from).apps
    executor.migrate(migrate_from)

    OldMenuCell = old_apps.get_model('data', 'MenuCell')
    old_cell = OldMenuCell.objects.create(
        uuid=str(uuid.uuid4()),
        page_id=page.id,
        order=1,
        root_page_id=page.id,
        initial_level=1,
        depth=2,
    )

    executor = MigrationExecutor(connection)
    executor.migrate(migrate_to)
    executor.loader.build_graph()

    cells = UniverCell.objects.all()
    assert len(cells) == 1
    assert cells[0].attributes == {'root_page': page.id, 'depth': 2, 'title': None, 'initial_level': 1}
    assert cells[0].class_name == 'combo.data.models.MenuCell'
    assert str(cells[0].uuid) == str(old_cell.uuid)


def test_dashboard_migration(transactional_db):
    page = Page.objects.create(title='One', slug='index')

    migrate_from = [('data', '0075_univercell'), ('dashboard', '0008_cell_uuid')]
    migrate_to = [
        ('data', '0077_convert_textcell_to_univercell'),
        ('dashboard', '0014_tile_remove_dashboard_temp_id'),
    ]

    executor = MigrationExecutor(connection)
    old_apps = executor.loader.project_state(migrate_from).apps
    executor.migrate(migrate_from)

    OldDashboardCell = old_apps.get_model('dashboard', 'DashboardCell')
    old_dashboard_cell = OldDashboardCell.objects.create(
        uuid=str(uuid.uuid4()),
        page_id=page.id,
        order=1,
    )

    OldTextCell = old_apps.get_model('data', 'TextCell')
    old_text_cell = OldTextCell.objects.create(
        uuid=str(uuid.uuid4()),
        page_id=page.id,
        order=1,
        text='hello',
    )

    user = get_user_model().objects.create(
        username='john.doe', first_name='John', last_name='Doe', email='john.doe@example.net'
    )
    Tile = old_apps.get_model('dashboard', 'Tile')
    tile = Tile.objects.create(
        dashboard=old_dashboard_cell,
        cell_type_id=ContentType.objects.get(model='oldtextcell').id,
        cell_pk=old_text_cell.id,
        order=1,
        user_id=user.id,
    )

    executor = MigrationExecutor(connection)
    executor.migrate(migrate_to)
    executor.loader.build_graph()

    cells = UniverCell.objects.all()
    assert len(cells) == 2
    assert cells[0].attributes == {}
    assert cells[0].class_name == 'combo.apps.dashboard.models.DashboardCell'
    assert str(cells[0].uuid) == str(old_dashboard_cell.uuid)

    migrated_tile = Tile.objects.get(id=tile.id)
    assert (
        migrated_tile.dashboard_id
        == UniverCell.objects.filter(class_name='combo.apps.dashboard.models.DashboardCell').first().id
    )
    assert migrated_tile.cell_type_id == ContentType.objects.get_for_model(TextCell).id
    assert (
        migrated_tile.cell_pk == UniverCell.objects.filter(class_name='combo.data.models.TextCell').first().id
    )


def test_gallery_migration(transactional_db):
    page = Page.objects.create(title='One', slug='index')

    migrate_from = [('data', '0075_univercell'), ('gallery', '0010_cell_uuid')]
    migrate_to = [
        ('data', '0077_convert_textcell_to_univercell'),
        ('gallery', '0016_image_remove_gallery_temp_id'),
    ]

    executor = MigrationExecutor(connection)
    old_apps = executor.loader.project_state(migrate_from).apps
    executor.migrate(migrate_from)

    OldGalleryCell = old_apps.get_model('gallery', 'GalleryCell')
    old_gallery_cell = OldGalleryCell.objects.create(
        uuid=str(uuid.uuid4()),
        page_id=page.id,
        order=1,
    )

    Image = old_apps.get_model('gallery', 'Image')
    image = Image.objects.create(
        gallery=old_gallery_cell,
        order=1,
        image='path/foo.jpeg',
        title='foo',
    )

    executor = MigrationExecutor(connection)
    executor.migrate(migrate_to)
    executor.loader.build_graph()

    cells = UniverCell.objects.all()
    assert len(cells) == 1
    assert cells[0].attributes == {'title': None}
    assert cells[0].class_name == 'combo.apps.gallery.models.GalleryCell'
    assert str(cells[0].uuid) == str(old_gallery_cell.uuid)

    migrated_image = Image.objects.get(id=image.id)
    assert (
        migrated_image.gallery_id
        == UniverCell.objects.filter(class_name='combo.apps.gallery.models.GalleryCell').first().id
    )
