import pytest
import requests
import responses
from django.db import connection
from django.db.migrations.executor import MigrationExecutor
from django.test.client import RequestFactory
from pyquery import PyQuery

from combo.apps.chrono.models import EventsCalendarCell
from combo.data.models import Page

from .test_public import login
from .utils import manager_submit_cell

pytestmark = pytest.mark.django_db


@pytest.fixture
def context():
    ctx = {
        'request': RequestFactory().get('/'),
        'synchronous': True,
    }
    ctx['request'].user = None
    ctx['request'].session = {}
    return ctx


EVENT_DATETIMES = {
    'data': [
        {
            'id': 'agenda-1@monday-8-sep-10-00-12-00',
            'label': 'Monday 8 Sep 10:00-12:00 (disabled)',
            'slug': 'monday-8-sep-10-00-12-00',
            'datetime': '2025-09-08 10:00:00',
            'end_datetime': '2025-09-08 12:00:00',
            'description': 'Foo bar',
            'pricing': '42€',
            'url': 'https://example.org',
            'places': {'full': False},
            'disabled': True,
        },
        {
            'id': 'agenda-1@wednesday-10-sep-10-00-12-00',
            'label': 'Wednesday 10 Sep 10:00-12:00 (bookable, red)',
            'slug': 'wednesday-10-sep-10-00-12-00',
            'datetime': '2025-09-10 10:00:00',
            'end_datetime': '2025-09-10 12:00:00',
            'description': 'Foo bar',
            'pricing': None,
            'url': None,
            'places': {'full': False},
            'disabled': False,
            'custom_field_color': '#FF0000',
        },
        {
            'id': 'agenda-1@wednesday-10-sep-14-00-15-00',
            'label': 'Wednesday 10 Sep 14:00-15:00 (full)',
            'slug': 'wednesday-10-sep-14-00-15-00',
            'datetime': '2025-09-10 14:00:00',
            'end_datetime': '2025-09-10 15:00:00',
            'description': 'Foo bar',
            'pricing': None,
            'url': None,
            'places': {'full': True},
            'disabled': True,
        },
        {
            'id': 'agenda-1@monday-15-sep-10-00-12-00',
            'label': 'Monday 15 Sep 10:00-12:00 (next week)',
            'slug': 'monday-15-sep-10-00-12-00',
            'datetime': '2025-09-15 10:00:00',
            'end_datetime': '2025-09-15 12:00:00',
            'description': 'Foo bar',
            'pricing': None,
            'url': None,
            'places': {'full': False},
            'disabled': True,
        },
    ]
}


@responses.activate
@pytest.mark.freeze_time('2025-09-23')
def test_events_calendar_cell(context):
    responses.get('http://chrono.example.org/api/agendas/datetimes/', json=EVENT_DATETIMES)

    page = Page.objects.create(title='Agenda', slug='index', template_name='standard')
    cell = EventsCalendarCell.objects.create(
        page=page, placeholder='content', order=0, agendas=['foo', 'bar'], booking_form_slug='my-form'
    )

    result = cell.render(context)

    assert len(responses.calls) == 1
    assert responses.calls[0].request.params['agendas'] == 'foo,bar'
    assert responses.calls[0].request.params['date_start'] == '2025-09-22'
    assert responses.calls[0].request.params['show_past_events'] == 'True'

    assert '--nb-days: 5;' in result
    assert '--nb-hours: 6;' in result
    assert '--hour-height: 80px;' in result

    pq = PyQuery(result)
    assert pq('.events-timetable--week-label').text() == 'Week from 8 to 12 September 2025'
    assert pq('.events-timetable--hours').text() == '10 a.m. 11 a.m. noon 1 p.m. 2 p.m. 3 p.m.'
    assert [(x.attrib['datetime'], x.text_content()) for x in pq('.events-timetable--day-label time')] == [
        ('2025-09-08', 'Mon 8 sep'),
        ('2025-09-09', 'Tue 9 sep'),
        ('2025-09-10', 'Wed 10 sep'),
        ('2025-09-11', 'Thu 11 sep'),
        ('2025-09-12', 'Fri 12 sep'),
    ]
    assert ['no-event' in x.attrib['class'] for x in pq('.events-timetable--day-label')] == [
        False,  # event on Monday
        True,
        False,  # events on Wednesday
        True,
        True,
    ]

    monday = pq('.events-timetable--day')[0]
    monday_events = pq(monday)('.events-timetable--event')
    assert len(monday_events) == 1

    event = pq(monday_events[0])
    assert event('.events-timetable--event-title').text() == 'Monday 8 Sep 10:00-12:00 (disabled)'
    assert event('.events-timetable--event-date').text() == '10 a.m. - noon'
    assert 'Book' not in event.text()

    assert '--event-top: calc(var(--hour-height) * 0.00' in event.attr('style')
    assert '--event-height: calc(var(--hour-height) * 2.00' in event.attr('style')
    assert '--event-color' not in event.attr('style')

    event_details = pq(monday)('.events-timetable--event-details')
    assert 'Foo bar' in event_details.text()
    assert '42€' in event_details.text()
    assert 'https://example.org' in event_details.text()

    tuesday = pq('.events-timetable--day')[1]
    tuesday_events = pq(tuesday)('.events-timetable--event')
    assert len(tuesday_events) == 0

    wednesday = pq('.events-timetable--day')[2]
    wednesday_events = pq(wednesday)('.events-timetable--event')
    assert len(wednesday_events) == 2

    event = pq(wednesday_events[0])
    assert event('.events-timetable--event-title').text() == 'Wednesday 10 Sep 10:00-12:00 (bookable, red)'
    assert event('.events-timetable--event-date').text() == '10 a.m. - noon'
    assert 'Book' in event.text()
    assert (
        event('.events-timetable--event-register').attr['href']
        == 'http://127.0.0.1:8999/my-form/?event=wednesday-10-sep-10-00-12-00&cancelurl=http://localhost/&agenda=agenda-1'
    )

    assert '--event-top: calc(var(--hour-height) * 0.00' in event.attr('style')
    assert '--event-height: calc(var(--hour-height) * 2.00' in event.attr('style')
    assert '--event-color: #FF0000' in event.attr('style')

    event = pq(wednesday_events[1])
    assert event('.events-timetable--event-title').text() == 'Wednesday 10 Sep 14:00-15:00 (full)'
    assert event('.events-timetable--event-date').text() == '2 p.m. - 3 p.m.'
    assert 'Book' not in event.text()
    assert 'Full' in event.text()

    assert '--event-top: calc(var(--hour-height) * 4.00' in event.attr('style')
    assert '--event-height: calc(var(--hour-height) * 1.00' in event.attr('style')
    assert '--event-color' not in event.attr('style')

    # check pagination
    assert pq('.events-timetable--change-week.previous').attr['disabled'] == 'disabled'
    assert pq('.events-timetable--change-week.next').attr['disabled'] is None

    next_week = pq('.events-timetable--change-week.next').attr['data-week']
    context['request'] = RequestFactory().get('/?week=%s' % next_week)

    result = cell.render(context)

    pq = PyQuery(result)
    assert pq('.events-timetable--change-week.previous').attr['disabled'] is None
    assert pq('.events-timetable--change-week.previous').attr['data-week'] == '2025-09-08'
    assert pq('.events-timetable--change-week.next').attr['disabled'] == 'disabled'

    assert pq('.events-timetable--week-label').text() == 'Week from 15 to 19 September 2025'

    assert len(pq('.events-timetable--event')) == 1
    assert pq('.events-timetable--event-title').text() == 'Monday 15 Sep 10:00-12:00 (next week)'


def test_events_calendar_cell_weekend(context, nocache):
    page = Page.objects.create(title='Agenda', slug='index', template_name='standard')
    cell = EventsCalendarCell.objects.create(
        page=page, placeholder='content', order=0, agendas=['foo', 'bar'], booking_form_slug='my-form'
    )

    event_datetimes = EVENT_DATETIMES.copy()

    with responses.RequestsMock() as rsps:
        rsps.get('http://chrono.example.org/api/agendas/datetimes/', json=event_datetimes)
        result = cell.render(context)

    pq = PyQuery(result)
    assert len(pq('.events-timetable--day-label time')) == 5
    assert pq('.events-timetable--day-label time')[-1].text_content() == 'Fri 12 sep'

    # event ends on Saturday
    event_datetimes['data'][2]['datetime'] = '2025-09-13 10:00:00'
    event_datetimes['data'][2]['end_datetime'] = '2025-09-13 12:00:00'

    with responses.RequestsMock() as rsps:
        rsps.get('http://chrono.example.org/api/agendas/datetimes/', json=event_datetimes)
        result = cell.render(context)

    pq = PyQuery(result)
    assert len(pq('.events-timetable--day-label time')) == 6
    assert pq('.events-timetable--day-label time')[-1].text_content() == 'Sat 13 sep'

    # event ends on Sunday
    event_datetimes['data'][2]['datetime'] = '2025-09-14 10:00:00'
    event_datetimes['data'][2]['end_datetime'] = '2025-09-14 12:00:00'

    with responses.RequestsMock() as rsps:
        rsps.get('http://chrono.example.org/api/agendas/datetimes/', json=event_datetimes)
        result = cell.render(context)

    pq = PyQuery(result)
    assert len(pq('.events-timetable--day-label time')) == 7
    assert pq('.events-timetable--day-label time')[-1].text_content() == 'Sun 14 sep'


@responses.activate
def test_events_calendar_cell_booking_form_slug(context):
    responses.get('http://chrono.example.org/api/agendas/datetimes/', json=EVENT_DATETIMES)

    page = Page.objects.create(title='Agenda', slug='index', template_name='standard')
    cell = EventsCalendarCell.objects.create(
        page=page, placeholder='content', order=0, agendas=['foo', 'bar'], booking_form_slug='my-form'
    )

    result = cell.render(context)
    assert 'Book' in result

    cell.booking_form_slug = ''
    cell.save()
    cell.refresh_from_db()

    result = cell.render(context)
    assert 'Book' not in result


def test_events_calendar_cell_api_error(context):
    page = Page.objects.create(title='Agenda', slug='index', template_name='standard')
    cell = EventsCalendarCell.objects.create(
        page=page, placeholder='content', order=0, agendas=['foo', 'bar'], booking_form_slug='my-form'
    )

    with responses.RequestsMock() as rsps:
        rsps.get('http://chrono.example.org/api/agendas/datetimes/', body=requests.RequestException())
        result = cell.render(context)

    pq = PyQuery(result)
    assert len(pq('.events-timetable--day-label time')) == 5
    assert len(pq('.events-timetable--event')) == 0

    with responses.RequestsMock() as rsps:
        rsps.get('http://chrono.example.org/api/agendas/datetimes/', status=403)
        result = cell.render(context)

    pq = PyQuery(result)
    assert len(pq('.events-timetable--day-label time')) == 5
    assert len(pq('.events-timetable--event')) == 0

    with responses.RequestsMock() as rsps:
        rsps.get('http://chrono.example.org/api/agendas/datetimes/', body='not-json')
        result = cell.render(context)

    pq = PyQuery(result)
    assert len(pq('.events-timetable--day-label time')) == 5
    assert len(pq('.events-timetable--event')) == 0


@responses.activate
@pytest.mark.freeze_time('2025-09-23')
def test_events_calendar_cell_start_on_half_hour(context):
    json_data = {
        'data': [
            {
                'id': 'agenda-1@monday-08-sep-10-30-12-00',
                'label': 'Monday 08 Sep 10:30-12:00',
                'slug': 'monday-08-sep-10-30-12-00',
                'datetime': '2025-09-08 10:30:00',
                'end_datetime': '2025-09-08 12:00:00',
                'description': 'Foo bar',
                'pricing': None,
                'url': None,
                'places': {'full': False},
                'disabled': False,
            },
        ]
    }
    responses.get('http://chrono.example.org/api/agendas/datetimes/', json=json_data)

    page = Page.objects.create(title='Agenda', slug='index', template_name='standard')
    cell = EventsCalendarCell.objects.create(page=page, placeholder='content', order=0, agendas='foo')

    result = cell.render(context)

    pq = PyQuery(result)
    assert pq('.events-timetable--hours').text() == '10 a.m. 11 a.m. noon'

    event = pq('.events-timetable--event')
    assert '--event-top: calc(var(--hour-height) * 0.50' in event.attr('style')


@responses.activate
def test_events_calendar_cell_categories(context):
    responses.get('http://chrono.example.org/api/agendas/datetimes/', json=EVENT_DATETIMES)

    page = Page.objects.create(title='Agenda', slug='index', template_name='standard')
    cell = EventsCalendarCell.objects.create(
        page=page, placeholder='content', order=0, agendas=['category:foo', 'bar']
    )

    cell.render(context)

    assert len(responses.calls) == 1
    assert responses.calls[0].request.params['categories'] == 'foo'
    assert responses.calls[0].request.params['agendas'] == 'bar'


@responses.activate
def test_events_calendar_cell_page_variables(app, admin_user):
    responses.get('http://chrono.example.org/api/agendas/datetimes/', json=EVENT_DATETIMES)

    Page.objects.create(title='One', slug='index')
    page = Page.objects.create(
        title='Agenda',
        slug='agenda',
        sub_slug='card_id',
        extra_variables={'foo': '{{ 40|add:card_id }}'},
    )
    EventsCalendarCell.objects.create(page=page, placeholder='content', order=0, agendas=['variable:foo'])

    app = login(app)
    app.get('/agenda/2/')

    assert len(responses.calls) == 1
    assert responses.calls[0].request.params['agendas'] == '42'

    # no crash if variable is missing
    page.extra_variables.clear()
    page.save()

    app.get('/agenda/2/')

    assert len(responses.calls) == 2
    assert responses.calls[1].request.params['agendas'] == 'variable:foo'


@responses.activate
def test_events_calendar_cell_manager(app, admin_user, nocache):
    agendas = [
        {
            'text': 'Agenda 1',
            'id': 'agenda-1',
            'category': 'category-a',
            'category_label': 'Category A',
        },
        {
            'text': 'Agenda 2',
            'id': 'agenda-2',
            'category': 'category-b',
            'category_label': 'Category B',
        },
        {
            'text': 'Agenda 3',
            'id': 'agenda-3',
            'category': None,
            'category_label': None,
        },
    ]
    api_url = 'http://chrono.example.org/api/agenda/'
    responses.get(api_url, json={'data': agendas})

    page = Page.objects.create(
        title='Agenda', slug='index', template_name='standard', extra_variables={'foo': '{{ bar }}'}
    )
    cell = EventsCalendarCell.objects.create(page=page, placeholder='content', order=0)

    app = login(app)
    resp = app.get('/manage/pages/%s/' % page.id)

    field_prefix = 'cchrono_eventscalendarcell-%s-' % cell.id
    assert resp.form[field_prefix + 'agendas'].options == [
        ('category:category-a', False, 'All agendas of category Category A'),
        ('agenda-1', False, 'Agenda 1'),
        ('category:category-b', False, 'All agendas of category Category B'),
        ('agenda-2', False, 'Agenda 2'),
        ('agenda-3', False, 'Agenda 3'),
        ('variable:foo', False, 'foo'),
    ]
    assert len(resp.pyquery('.cell optgroup')) == 4

    # check first option of each groups
    assert resp.pyquery('.cell optgroup[label="Category A"] option').val() == 'category:category-a'
    assert resp.pyquery('.cell optgroup[label="Category B"] option').val() == 'category:category-b'
    assert resp.pyquery('.cell optgroup[label="Misc"] option').val() == 'agenda-3'
    assert resp.pyquery('.cell optgroup[label="Page variables"] option').val() == 'variable:foo'

    resp.form[field_prefix + 'agendas'].select_multiple(
        texts=['All agendas of category Category A', 'Agenda 3']
    )
    manager_submit_cell(resp.form)

    cell.refresh_from_db()
    assert cell.agendas == ['category:category-a', 'agenda-3']

    page.extra_variables.clear()
    page.save()

    # no option group when no agenda with category
    agendas = [
        {
            'text': 'Agenda 1',
            'id': 'agenda-1',
            'category': None,
            'category_label': None,
        },
    ]
    responses.replace(responses.GET, api_url, json={'data': agendas})

    resp = app.get('/manage/pages/%s/' % page.id)
    assert resp.form[field_prefix + 'agendas'].options == [
        ('agenda-1', False, 'Agenda 1'),
    ]
    assert len(resp.pyquery('.cell optgroup')) == 0

    # no agendas
    responses.replace(responses.GET, api_url, json={'data': []})

    resp = app.get('/manage/pages/%s/' % page.id)
    assert len(resp.form[field_prefix + 'agendas'].options) == 0


def test_events_calendar_cell_migrate_agenda_field():
    page = Page.objects.create(title='One', slug='index')
    cell = EventsCalendarCell.objects.create(page=page, placeholder='content', order=0, agendas='foo,bar')
    cell2 = EventsCalendarCell.objects.create(page=page, placeholder='content', order=1, agendas=['a', 'b'])

    migrate_from = [('chrono', '0001_initial')]
    migrate_to = [('chrono', '0002_migrate_agenda_field')]
    executor = MigrationExecutor(connection)
    executor.migrate(migrate_from)

    executor = MigrationExecutor(connection)
    executor.migrate(migrate_to)

    cell.refresh_from_db()
    assert cell.agendas == ['foo', 'bar']

    cell2.refresh_from_db()
    assert cell2.agendas == ['a', 'b']
