# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import uuid

from django import forms
from django.forms.widgets import Select
from django.utils.translation import gettext_lazy as _
from django.utils.translation import pgettext_lazy

from ..models.layouts import CellLayout
from ..utils.layouts import get_block_class_by_type
from .data_sources import DataSourceField


class LayoutSelectionWidget(Select):
    widget_template_name = 'combo/layouts/layout_selection_widget.html'


class LayoutTypeWidget(forms.RadioSelect):
    widget_template_name = 'combo/layouts/layout_type_widget.html'


class LayoutAddForm(forms.ModelForm):
    mode = forms.ChoiceField(
        label=_('Mode'),
        choices=(
            ('card', pgettext_lazy('card-display-mode', 'Card')),
            ('table', pgettext_lazy('card-display-mode', 'Table')),
        ),
        initial='card',
        required=True,
        widget=LayoutTypeWidget,
    )

    class Meta:
        model = CellLayout
        fields = ('title', '_data_source')
        field_classes = {
            '_data_source': DataSourceField,
        }

    def save(self, *args, **kwargs):
        mode_params = {
            'card': {'root-type': 'card-root', 'child-type': 'stack'},
            'table': {'root-type': 'table-root', 'child-type': 'table-columns'},
        }[self.cleaned_data['mode']]
        self.instance._layout = {
            'type': mode_params['root-type'],
            'children': [{'type': mode_params['child-type'], 'uuid': str(uuid.uuid4()), 'children': []}],
        }
        return super().save(*args, **kwargs)


class BlockTypeWidget(forms.RadioSelect):
    widget_template_name = 'combo/layouts/block_type_widget.html'

    def get_options(self):
        return {x[0]: get_block_class_by_type(x[0]) for x in self.choices}


class LayoutAddBlockForm(forms.Form):
    uuid = forms.CharField(required=True, widget=forms.HiddenInput)
    type = forms.ChoiceField(
        label='',
        choices=(
            # translations will come from the matching classes
            ('group', 'Group'),
            ('row', 'Row'),
            ('stack', 'Stack'),
            ('attribute', 'Attribute'),
            ('text', 'Text'),
        ),
        required=True,
        widget=BlockTypeWidget,
    )


class LayoutDeleteBlockForm(forms.Form):
    uuid = forms.CharField(required=True, widget=forms.HiddenInput)


class LayoutBlockContainerOptions(forms.Form):
    uuid = forms.CharField(required=True, widget=forms.HiddenInput)
    type = forms.ChoiceField(
        label=_('Block type'),
        choices=(
            ('group', _('Group')),
            ('row', _('Row')),
            ('stack', _('Stack')),
        ),
        required=True,
        widget=forms.RadioSelect,
    )
    justification = forms.ChoiceField(
        label=_('Justification'),
        choices=(
            ('left', _('Justify items left')),
            ('center', _('Justify items center')),
            ('right', _('Justify items right')),
            ('space-between', _('Space between items')),
        ),
        required=True,
        widget=forms.RadioSelect,
        initial='left',
    )
    allow_wrap = forms.BooleanField(
        label=_('Allow to wrap to multiple lines'),
        required=False,
    )
    link_to_item = forms.ChoiceField(
        label=_('Link to item'),
        choices=(),
        required=False,
    )

    def __init__(self, *args, **kwargs):
        self.layout_block = kwargs.pop('layout_block')
        self.data_source = kwargs.pop('data_source')
        super().__init__(*args, **kwargs)
        self.fields.update(self.layout_block.parent.get_fields_for_child_form())
        self.fields['link_to_item'].choices = [
            (None, _('No link'))
        ] + self.data_source.get_link_to_item_choices()
        # link_to_item widget could be changed to HiddenInput if there were no choices
        # but for now all datasources return at least one choice.
        field_order = []
        field_order = [x for x, y in self.fields.items() if getattr(y, 'order_start', False)] + [
            x for x, y in self.fields.items() if getattr(y, 'order_start', True)
        ]
        self.order_fields(field_order)


class LayoutBlockAttributeOptions(forms.Form):
    uuid = forms.CharField(required=True, widget=forms.HiddenInput)
    attribute = forms.CharField(label=_('Attribute'), required=True, widget=forms.Select)
    style = forms.ChoiceField(
        label=_('Style'),
        required=True,
        widget=forms.Select,
        choices=(('text', _('Simple text')), ('title', _('Title')), ('subtitle', _('Subtitle'))),
        initial='text',
    )
    label_display = forms.ChoiceField(
        label=_('Display label'),
        required=True,
        widget=forms.Select,
        choices=(('no', _('None - value only')), ('above', _('Above value')), ('before', _('Left of value'))),
        initial='no',
    )
    empty_behaviour = forms.ChoiceField(
        label=_('Behaviour for empty content'),
        required=True,
        widget=forms.Select,
        choices=(('hide', _('Do not include')), ('text', _('Display custom text'))),
        initial='hide',
    )
    text_if_empty = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.layout_block = kwargs.pop('layout_block')
        self.data_source = kwargs.pop('data_source')
        super().__init__(*args, **kwargs)
        if self.initial.get('attribute'):
            self.fields.update(self.data_source.get_form_fields_for_attribute(self.initial.get('attribute')))
        self.fields.update(self.layout_block.parent.get_fields_for_child_form())
        self.fields['attribute'].widget.choices = (
            self.layout_block.get_cell_layout().data_source.get_attribute_choices()
        )
        self.fields['empty_behaviour'].widget.attrs['data-dynamic-display-parent'] = 'true'
        self.fields['text_if_empty'].widget.container_attrs = {
            'data-dynamic-display-child-of': 'empty_behaviour',
            'data-dynamic-display-value': 'text',
        }
        self.fields['style'].widget.attrs['data-dynamic-display-parent'] = 'true'
        self.fields['label_display'].widget.container_attrs = {
            'data-dynamic-display-child-of': 'style',
            'data-dynamic-display-value': 'text',
        }


class LayoutBlockTextOptions(forms.Form):
    uuid = forms.CharField(required=True, widget=forms.HiddenInput)
    text = forms.CharField(label=_('Text'), required=False, widget=forms.Textarea(attrs={'rows': 5}))
    style = forms.ChoiceField(
        label=_('Style'),
        required=True,
        widget=forms.Select,
        choices=(('text', _('Simple text')), ('title', _('Title')), ('subtitle', _('Subtitle'))),
        initial='text',
    )

    def __init__(self, *args, **kwargs):
        self.layout_block = kwargs.pop('layout_block')
        self.data_source = kwargs.pop('data_source')
        super().__init__(*args, **kwargs)
        self.fields.update(self.layout_block.parent.get_fields_for_child_form())


class LayoutCardRootOptions(forms.Form):
    uuid = forms.CharField(required=True, widget=forms.HiddenInput)
    card_title_mode = forms.ChoiceField(
        label=_('Top Card Title'),
        required=True,
        choices=[
            ('none', _('No title')),
            ('auto', _('Automatic title from content')),
            ('custom', _('Custom title from template')),
        ],
    )
    card_title_template = forms.CharField(
        label=_('Title template'),
        required=False,
    )
    inner_card_title_levels = forms.ChoiceField(
        label=_('Level of titles inside the card'),
        required=False,
        choices=[
            ('h1', _('Level 1')),
            ('h2', _('Level 2')),
            ('h3', _('Level 3')),
            ('h4', _('Level 4')),
        ],
        initial='h3',
    )

    def __init__(self, *args, **kwargs):
        self.layout_block = kwargs.pop('layout_block')
        self.data_source = kwargs.pop('data_source')
        super().__init__(*args, **kwargs)
        self.fields['card_title_mode'].widget.attrs['data-dynamic-display-parent'] = 'true'
        self.fields['card_title_template'].widget.container_attrs = {
            'data-dynamic-display-child-of': 'card_title_mode',
            'data-dynamic-display-value': 'custom',
        }
