# combo - content management system
# Copyright (C) 2014-2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.core.paginator import Paginator
from django.utils.translation import gettext_lazy as _


def fill_pagination_context(context, count, paginate_by, offset):
    context['pagination_links'] = pagination_links = []

    if not isinstance(count, int) or count <= paginate_by:
        return

    class ObjectList:
        # fake object to avoid "yet another .get_elided_page_range()"
        def count(self):
            return count

    paginator = Paginator(ObjectList(), per_page=paginate_by)
    current_page = offset // paginate_by + 1
    pages = list(paginator.get_elided_page_range(current_page, on_each_side=3, on_ends=1))
    if pages:
        # previous link
        pagination_links.append(
            {
                'label': '<',
                'arialabel': _('Previous page'),
                'offset': offset - paginate_by,
                'disabled': current_page < 2,
                'active': False,
            },
        )

        # page links and ellipsis
        for page in pages:
            if isinstance(page, int):
                pagination_links.append(
                    {
                        'label': page,
                        'arialabel': _('Page %s') % page,
                        'offset': (page - 1) * paginate_by,
                        'disabled': current_page == page,
                        'active': current_page == page,
                    }
                )
            else:
                # ellipsis
                pagination_links.append({'label': page})

        # next link
        pagination_links.append(
            {
                'label': '>',
                'arialabel': _('Next page'),
                'offset': offset + paginate_by,
                'disabled': current_page >= paginator.num_pages,
                'active': False,
            }
        )

    context['pagination_state'] = {
        'count': count,
        'first': offset + 1,
        'last': min(offset + paginate_by, count),
    }
