# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import dataclasses

from django import forms
from django.conf import settings
from django.utils.translation import gettext_lazy as _

from combo import utils
from combo.data.library import register_cell_class
from combo.data.models import UniverCell
from combo.utils.pagination import fill_pagination_context

from ..utils.data_sources import WithDataSourceMixin
from .layouts import CellLayout


@register_cell_class
class ContentCell(UniverCell, WithDataSourceMixin):
    default_template_name = 'combo/content/cell.html'
    manager_form_template = 'combo/content/manager_cell_form.html'

    user_dependant = True

    class Meta:
        proxy = True
        verbose_name = _('Content')

    class Media:
        js = ('js/combo.layout.pagination.js',)
        css = {'all': ('css/combo.layouts.css',)}

    @classmethod
    def is_enabled(cls):
        return settings.CONTENT_CELL_ENABLED

    def is_visible(self, *args, **kwargs):
        return bool(self.data_source) and super().is_visible(*args, **kwargs)

    def get_additional_label(self):
        parts = []
        if self.title:
            parts.append(self.title)
        if self.data_source:
            parts.append(self.data_source.label)
        if not parts:
            return None
        return utils.ellipsize(' - '.join(parts))

    def get_dependencies(self):
        yield from super().get_dependencies()
        layout = self.get_layout()
        if layout and layout.id:  # not built-in __list__
            yield layout
        if self.data_source:
            yield from self.data_source.get_dependencies()

    @classmethod
    def get_attribute_fields(cls):
        from ..forms import DataSourceField

        return {
            '_data_source': DataSourceField(label=_('Data source')),
            'layout': forms.ChoiceField(label=_('Layout'), required=False),
            'paginate_by': forms.IntegerField(
                label=_('Number of items per page (default 10, 0 to disable)'),
                min_value=0,
                required=False,
                initial=10,
            ),
            'title': forms.CharField(label=_('Title'), required=False),
        }

    _layout_object = None

    def get_layout(self):
        if self._layout_object:
            return self._layout_object

        if not self.layout or self.layout == '__list__':
            self._layout_object = CellLayout()
            self._layout_object._layout = {
                'type': 'list-root',
                'children': [
                    {'type': 'attribute', 'attributes': {'attribute': 'metadata:text', 'link_to_item': True}}
                ],
            }
        else:
            self._layout_object = CellLayout.objects.get(uuid=self.layout)

        return self._layout_object

    @property
    def container_cell_class(self):
        if self.get_layout().layout.type in ('table-root', 'list-root'):
            return 'cell'
        return ''

    def get_default_form_class(self):
        from ..forms.cells import ContentCellForm

        form_class = super().get_default_form_class()
        return type(f'{self.class_name}AutoForm', (ContentCellForm, form_class), {})

    def get_cell_extra_context(self, context):
        if context.get('placeholder_search_mode'):
            return {}

        try:
            page_offset = int(context['request'].GET.get('page_offset'))
        except (TypeError, ValueError):
            page_offset = 0

        context['content_entries'] = self.data_source.get_data(
            context=context, paginate_by=self.paginate_by, page_offset=page_offset
        )

        count = context['content_entries'].get('count') or 0
        fill_pagination_context(context, count, self.paginate_by or 10, page_offset)

        return super().get_cell_extra_context(context)

    def save(self, *args, **kwargs):
        if self._data_source:
            self.data_source.refresh_cache()
            self._data_source = dataclasses.asdict(self.data_source)
        return super().save(*args, **kwargs)
