# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django import forms
from django.template import Context, Template
from django.utils.html import format_html
from django.utils.safestring import mark_safe
from django.utils.translation import gettext_lazy as _
from django.utils.translation import pgettext_lazy


class LayoutBlock:
    uuid = None
    type = None
    children = None
    attributes = None
    parent = None

    css_class_attributes = []
    css_child_class_attributes = []

    is_container = False

    def __init__(self, type=None, uuid=None, parent=None, children=None, attributes=None):
        self.type = type
        self.uuid = uuid
        self.children = children or []
        self.attributes = attributes or {}
        self.parent = parent

    def must_be_displayed(self):
        return True

    def get_block_by_uuid(self, uuid):
        if uuid == 'root' and isinstance(self, LayoutBlockRoot):
            return self

        def traverse_to_uuid(parent):
            for block in parent.children or []:
                if block.uuid == uuid:
                    yield block
                else:
                    yield from traverse_to_uuid(parent=block)

        block = list(traverse_to_uuid(self))
        return block[0] if block else None

    def get_parent_by_uuid(self, parent_uuid):

        def traverse_to_uuid(parent):
            for block in parent.children or []:
                if block.uuid == parent_uuid:
                    yield (parent, block)
                else:
                    yield from traverse_to_uuid(parent=block)

        block = list(traverse_to_uuid(self))
        return block[0][0] if block else None

    def add_block(self, new_block, root_uuid):
        root_block = self.get_block_by_uuid(root_uuid)
        root_block.children.append(new_block)

    def delete_block(self, uuid):
        parent_block = self.get_parent_by_uuid(uuid)
        parent_block.children = [x for x in parent_block.children if x.uuid != uuid]

    def as_json(self):
        return {
            'type': self.type,
            'uuid': self.uuid,
            'children': [x.as_json() for x in self.children],
            'attributes': self.attributes,
        }

    def get_root(self):
        return self.parent.get_root() if self.parent else self

    def get_options_form_class(self):
        return None

    def get_cell_layout(self):
        return self.get_root().cell_layout

    def get_fields_for_child_form(self):
        return {}

    def get_css_classes(self):
        classes = [f'cb-type-{self.type}']
        for k, v in (self.attributes or {}).items():
            if k in self.css_class_attributes or (
                self.parent and k in self.parent.css_child_class_attributes
            ):
                if v is True:
                    classes.append(f'cb-{k}')
                elif v:
                    classes.append(f'cb-{k}-{v}')
        return ' '.join(classes)

    def get_css_child_style(self, child):
        return ''

    def get_css_style(self):
        return ''.join([self.parent.get_css_child_style(child=self) if self.parent else '']).strip()

    def get_outer_tag(self):
        return mark_safe(f'<div class="{self.get_css_classes()}" style="{self.get_css_style()}">')

    def get_outer_close_tag(self):
        return mark_safe('</div>')

    def get_inner_tag(self):
        return ''

    def get_inner_close_tag(self):
        return ''


class LayoutBlockContainer(LayoutBlock):
    is_container = True

    def get_options_form_class(self):
        from ..forms.layouts import LayoutBlockContainerOptions

        return LayoutBlockContainerOptions


class LayoutBlockRoot(LayoutBlockContainer):
    include_cell_body_div = True

    def get_manager_content_before_layout(self):
        return ''

    def get_base_title_level(self):
        return 3  # cell title would be <h2>


class LayoutBlockCardRoot(LayoutBlockRoot):
    def get_manager_content_before_layout(self):
        if self.attributes.get('card_title_mode') in ('auto', 'custom'):
            if self.attributes.get('card_title_mode') == 'auto':
                title = _('Automatic title from content')
            else:
                title = _('Custom title from template')
            return format_html('<div class="content-before-layout"><h3>{}</h3></div>', title)
        return ''

    def get_inner_tag(self):
        parts = ['<div class="cell">']
        self.inner_close_tag = '</div>'
        if self.attributes.get('card_title_mode') in ('auto', 'custom'):
            if self.attributes.get('card_title_mode') == 'auto':
                title = self.get_root().bound_content.title()
            else:
                # custom
                template = self.attributes.get('card_title_template') or ''
                ctx = Context({'data': self.get_root().bound_content})
                title = Template(template).render(ctx)
            parts.append(format_html('<h2 class="cell--title">{}</h2>', title))
            parts.append('<div class="cell--body">')
            self.inner_close_tag = '</div></div>'
        return mark_safe(''.join(parts))

    def get_inner_close_tag(self):
        return mark_safe(self.inner_close_tag)

    def get_base_title_level(self):
        if self.children:
            level = self.attributes.get('inner_card_title_levels', 'h3')
            return ['h1', 'h2', 'h3', 'h4'].index(level) + 1
        return super().get_base_title_level()

    def get_options_form_class(self):
        from ..forms.layouts import LayoutCardRootOptions

        return LayoutCardRootOptions


class LayoutBlockTableRoot(LayoutBlockRoot):
    def get_outer_tag(self):
        parts = ['<div class="pk-table-wrapper"><table class="pk-data-table">']
        labels = [x.attributes.get('column_label', '') for x in self.children[0].children]
        if any(labels):
            parts.append('<thead><tr>')
            for label in labels:
                parts.append(format_html('<th>{}</th>', label))
            parts.append('</tr></thead>')
        parts.append('<tbody>')
        return mark_safe(''.join(parts))

    def get_outer_close_tag(self):
        return mark_safe('</tbody></table></div>')


class LayoutBlockListRoot(LayoutBlockRoot):
    include_cell_body_div = False

    def get_outer_tag(self):
        return mark_safe('<div class="links-list"><ul>')

    def get_outer_close_tag(self):
        return mark_safe('</ul></div>')

    def get_inner_tag(self):
        return mark_safe('<li>')

    def get_inner_close_tag(self):
        return mark_safe('</li>')


class LayoutBlockGroup(LayoutBlockContainer):
    label = pgettext_lazy('layout-block', 'Group')
    description = _('Gather blocks in a container.')


class LayoutBlockFlexContainer(LayoutBlockContainer):
    css_class_attributes = ['justification', 'allow_wrap']
    css_child_class_attributes = ['sizing']

    def get_css_child_style(self, child):
        if child.attributes.get('sizing') == 'fixed' and child.attributes.get('fixed_size'):
            return f'flex-basis: {child.attributes["fixed_size"]};'
        return ''


class LayoutBlockRow(LayoutBlockFlexContainer):
    label = pgettext_lazy('layout-block', 'Row')
    description = _('Arrange blocks horizontally.')

    def get_fields_for_child_form(self):
        fields = {
            'sizing': forms.ChoiceField(
                label=_('Width'),
                required=True,
                choices=[
                    ('fit', _('Fit contents')),
                    ('grow', _('Stretch to fill available space')),
                    ('fixed', _('Specify a fixed width')),
                ],
                widget=forms.RadioSelect,
                initial='fit',
            ),
            'fixed_size': forms.CharField(
                label=_('Fixed width'),
                max_length=50,
                required=False,
            ),
        }
        fields['sizing'].widget.attrs['data-dynamic-display-parent'] = 'true'
        fields['fixed_size'].widget.container_attrs = {
            'data-dynamic-display-child-of': 'sizing',
            'data-dynamic-display-value': 'fixed',
        }
        return fields


class LayoutBlockStack(LayoutBlockFlexContainer):
    label = pgettext_lazy('layout-block', 'Stack')
    description = _('Arrange blocks vertically.')

    def get_fields_for_child_form(self):
        fields = {
            'sizing': forms.ChoiceField(
                label=_('Height'),
                required=True,
                choices=[
                    ('fit', _('Fit contents')),
                    ('grow', _('Stretch to fill available space')),
                    ('fixed', _('Specify a fixed height')),
                ],
                widget=forms.RadioSelect,
                initial='fit',
            ),
            'fixed_size': forms.CharField(
                label=_('Fixed height'),
                max_length=50,
                required=False,
            ),
        }
        fields['sizing'].widget.attrs['data-dynamic-display-parent'] = 'true'
        fields['fixed_size'].widget.container_attrs = {
            'data-dynamic-display-child-of': 'sizing',
            'data-dynamic-display-value': 'fixed',
        }
        return fields


class LayoutBlockAttribute(LayoutBlock):
    label = pgettext_lazy('layout-block', 'Attribute')
    description = _('Display data from content.')

    def get_options_form_class(self):
        from ..forms.layouts import LayoutBlockAttributeOptions

        return LayoutBlockAttributeOptions

    def must_be_displayed(self):
        if not self.attributes.get('attribute'):  # not configured yet
            return False
        value = self.get_root().bound_content.get_value(self.attributes.get('attribute'), layout_block=self)
        if not (value or value is False) and self.attributes.get('empty_behaviour', 'hide') == 'hide':
            return False
        return True

    def get_content(self, context=None, placeholder=True):
        if not self.attributes.get('attribute'):
            return format_html('<em>{}</em>', _('unconfigured'))

        if self.attributes.get('style', 'text') != 'text':
            self.attributes['label_display'] = 'no'

        if placeholder or self.attributes.get('label_display') in ('above', 'before'):
            attribute_label = (
                self.get_cell_layout().data_source.get_attribute_label(self.attributes.get('attribute')) or ''
            )

        if placeholder:
            value = format_html('<em>{}</em>', attribute_label)
        else:
            value = self.get_root().bound_content.get_value(
                self.attributes.get('attribute'),
                layout_block=self,
                context=context,
            )
            if not (value or value is False):
                value = self.attributes.get('text_if_empty') or ''

        if self.attributes.get('label_display') == 'above':
            return format_html('<div class="field-label">{}</div><div>{}</div>', attribute_label, value)
        elif self.attributes.get('label_display') == 'before':
            return format_html(
                '<span class="field-label">{}</span><span class="field-separator">{}</span> <span>{}</span>',
                attribute_label,
                _(':'),
                value,
            )

        if placeholder:
            title_level = 3
        else:
            title_level = self.get_root().get_base_title_level()

        if self.attributes.get('style') == 'title':
            return format_html('<h{}>{}</h{}>', title_level, value, title_level)
        elif self.attributes.get('style') == 'subtitle':
            return format_html('<h{}>{}</h{}>', title_level + 1, value, title_level + 1)
        return value

    def render_content(self, context):
        return self.get_content(context=context, placeholder=False)


class LayoutBlockText(LayoutBlock):
    label = pgettext_lazy('layout-block', 'Text')
    description = _('Display text, possibly using a template.')

    def get_options_form_class(self):
        from ..forms.layouts import LayoutBlockTextOptions

        return LayoutBlockTextOptions

    def must_be_displayed(self):
        return bool(self.attributes.get('text'))

    def get_content(self, context=None, placeholder=True):
        if not self.attributes.get('text'):
            return format_html('<em>{}</em>', _('unconfigured'))

        value = self.attributes.get('text')

        if placeholder:
            title_level = 3
        else:
            ctx = Context({'data': self.get_root().bound_content})
            value = Template(value).render(ctx)
            title_level = self.get_root().get_base_title_level()

        if self.attributes.get('style') == 'title':
            return format_html('<h{}>{}</h{}>', title_level, value, title_level)
        elif self.attributes.get('style') == 'subtitle':
            return format_html('<h{}>{}</h{}>', title_level + 1, value, title_level + 1)
        return value

    def render_content(self, context):
        return self.get_content(context=context, placeholder=False)


class LayoutBlockTableColumns(LayoutBlockContainer):
    def get_outer_tag(self):
        return mark_safe('<tr>')

    def get_outer_close_tag(self):
        return mark_safe('</tr>')

    def get_inner_tag(self):
        return mark_safe('<td>')

    def get_inner_close_tag(self):
        return mark_safe('</td>')

    def get_options_form_class(self):
        return None

    def get_fields_for_child_form(self):
        fields = {
            'column_label': forms.CharField(label=_('Column label'), required=False),
            'label_display': forms.CharField(widget=forms.HiddenInput, initial='no'),
        }
        return fields


def get_block_class_by_type(type):
    return {
        'root': LayoutBlockRoot,
        'card-root': LayoutBlockCardRoot,
        'table-root': LayoutBlockTableRoot,
        'list-root': LayoutBlockListRoot,
        'table-columns': LayoutBlockTableColumns,
        'group': LayoutBlockGroup,
        'row': LayoutBlockRow,
        'stack': LayoutBlockStack,
        'attribute': LayoutBlockAttribute,
        'text': LayoutBlockText,
    }[type]


def create_block_from_json_data(data):
    def transform(block, parent=None):
        block_object = get_block_class_by_type(block.get('type'))(
            uuid=block.get('uuid'),
            type=block.get('type'),
            attributes=block.get('attributes') or {},
            parent=parent,
        )
        block_object.children = [transform(x, parent=block_object) for x in block.get('children') or []]
        return block_object

    return transform(data)
