# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import dataclasses

from django.apps import apps


class DataSourceData:
    def __init__(self, data_source, data):
        self._data = data
        self.data_source = data_source

        obj = self

        class MetadataAccessor:
            def __getattr__(self, attribute_name):
                return obj.get_value(f'metadata:{attribute_name}')

        class FieldsAccessor:
            def __getattr__(self, attribute_name):
                return obj.get_value(attribute_name)

        self.metadata = MetadataAccessor()
        self.fields = FieldsAccessor()

    def url(self, context=None, layout=None):
        link_to_item_choice = layout.attributes.get('link_to_item') if layout else True
        if link_to_item_choice is True:  # list mode, default value
            link_to_item_choice = self.data_source.get_link_to_item_choices()[0][0]
        return self._data.get(link_to_item_choice)


@dataclasses.dataclass
class DataSource:
    provider_id: str
    service_id: str
    slug: str
    label: str
    cache: dict = None

    def get_option_str(self):
        return f'{self.provider_id}:{self.service_id}:{self.slug}'

    @classmethod
    def get_by_option_str(cls, option_str):
        provider_id = option_str.split(':', 1)[0]
        for appconfig in apps.get_app_configs():
            if not hasattr(appconfig, 'get_data_source_providers'):
                continue
            for data_source_provider in appconfig.get_data_source_providers():
                if data_source_provider.provider_id == provider_id:
                    for data_source in data_source_provider.choices:
                        if data_source.get_option_str() == option_str:
                            return data_source

    def refresh_cache(self):
        pass

    def get_attribute_choices(self):
        return []  # pragma: no cover

    def get_link_to_item_choices(self):
        return []  # pragma: no cover

    def get_attribute_label(self, attribute_id):
        return [x[1] for x in self.get_attribute_choices() if x[0] == attribute_id][0]

    def get_form_fields_for_attribute(self, attribute_name):
        return {}

    def get_dependencies(self):
        return []


class DataSourceProvider:
    data_source_class = DataSource

    provider_id: str
    title: str
    choices: list

    @classmethod
    def get_all(cls):
        for appconfig in apps.get_app_configs():
            if not hasattr(appconfig, 'get_data_source_providers'):
                continue
            yield from appconfig.get_data_source_providers()

    @classmethod
    def get_by_provider_id(cls, provider_id):
        for provider in cls.get_all():
            if provider.provider_id == provider_id:
                return provider


def get_data_source_from_dict(value):
    data_source_object = None
    if value:
        provider = DataSourceProvider.get_by_provider_id(value['provider_id'])
        data_source_object = provider.data_source_class(**value)
        data_source_object.provider = provider
    return data_source_object


class WithDataSourceMixin:
    _data_source_object = None

    @property
    def data_source(self):
        if self._data_source_object:
            return self._data_source_object
        self._data_source_object = get_data_source_from_dict(self._data_source)
        return self._data_source_object

    @data_source.setter
    def data_source(self, value):
        self._data_source_object = value
        self._data_source = dataclasses.asdict(self.data_source)
