import uuid
from unittest import mock

import pytest

from combo.apps.content.models import CellLayout, ContentCell
from combo.apps.wcs.data_sources import CardsDataSource, UserFormsDataSource
from combo.data.models import Page

from .test_manager import login
from .wcs.utils import mocked_requests_send

pytestmark = pytest.mark.django_db


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_content_cell(mock_send, app, admin_user):
    layout = CellLayout.objects.create(title='Layout')
    layout.data_source = UserFormsDataSource(
        provider_id='wcs_user_forms', service_id='default', slug='__user_forms', label='test: User Forms'
    )
    layout.save()
    assert str(layout) == 'Layout'

    page = Page.objects.create(title='One', slug='one', template_name='standard')
    cell = ContentCell.objects.create(page=page, order=0, placeholder='content')

    app = login(app)
    resp = app.get('/one/')
    assert resp.pyquery('.content-cell').length == 0

    cell._data_source = layout._data_source
    cell.save()
    resp = app.get('/one/')
    url = resp.pyquery('.content-cell[data-ajax-cell-url]').attr('data-ajax-cell-url')
    resp = app.get(url)
    assert resp.pyquery('li').length == 3
    assert resp.pyquery('li:first-child').text() == 'Title 1'


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_content_cell_table(mock_send, app, admin_user):
    layout = CellLayout.objects.create(title='Layout')
    layout.data_source = UserFormsDataSource(
        provider_id='wcs_user_forms', service_id='default', slug='__user_forms', label='test: User Forms'
    )
    layout._layout = {
        'type': 'table-root',
        'uuid': None,
        'children': [
            {
                'type': 'table-columns',
                'uuid': str(uuid.uuid4()),
                'children': [
                    {
                        'type': 'attribute',
                        'uuid': str(uuid.uuid4()),
                        'children': [],
                        'attributes': {
                            'style': 'text',
                            'attribute': 'form_display_name',
                            'column_label': 'Label',
                            'label_display': 'no',
                            'text_if_empty': '',
                            'empty_behaviour': 'hide',
                        },
                    },
                ],
            }
        ],
    }
    layout.save()

    page = Page.objects.create(title='One', slug='one', template_name='standard')
    cell = ContentCell.objects.create(page=page, order=0, placeholder='content')
    cell._data_source = layout._data_source
    cell.attributes['layout'] = str(layout.uuid)
    cell.save()

    app = login(app)
    resp = app.get('/one/')
    url = resp.pyquery('.content-cell[data-ajax-cell-url]').attr('data-ajax-cell-url')
    resp = app.get(url)
    assert resp.pyquery('table thead tr').length == 1
    assert resp.pyquery('table tbody tr').length == 3
    assert resp.pyquery('table tbody tr:first-child td').length == 1
    assert resp.pyquery('table tbody tr:first-child td').text() == 'Title 1'

    # without table header
    layout.layout.children[0].children[0].attributes['column_label'] = ''
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('table thead tr').length == 0


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_content_cell_card(mock_send, app, admin_user):
    layout = CellLayout.objects.create(title='Layout')
    layout.data_source = UserFormsDataSource(
        provider_id='wcs_user_forms', service_id='default', slug='__user_forms', label='test: User Forms'
    )
    layout._layout = {
        'type': 'card-root',
        'uuid': None,
        'children': [
            {
                'type': 'stack',
                'uuid': str(uuid.uuid4()),
                'children': [
                    {
                        'type': 'attribute',
                        'uuid': str(uuid.uuid4()),
                        'children': [],
                        'attributes': {
                            'style': 'text',
                            'attribute': 'form_display_name',
                            'column_label': 'Label',
                            'label_display': 'no',
                            'text_if_empty': '',
                            'empty_behaviour': 'hide',
                        },
                    },
                    {
                        'type': 'text',
                        'uuid': str(uuid.uuid4()),
                        'children': [],
                        'attributes': {
                            'style': 'text',
                            'text': 'hello',
                        },
                    },
                ],
            }
        ],
    }
    layout.save()

    page = Page.objects.create(title='One', slug='one', template_name='standard')
    cell = ContentCell.objects.create(page=page, order=0, placeholder='content')
    cell._data_source = layout._data_source
    cell.attributes['layout'] = str(layout.uuid)
    cell.save()

    app = login(app)
    resp = app.get('/one/')
    url = resp.pyquery('.content-cell[data-ajax-cell-url]').attr('data-ajax-cell-url')
    resp = app.get(url)
    assert resp.pyquery('.cell').length == 3

    # fixed size
    layout.layout.children[0].children[0].attributes['sizing'] = 'fixed'
    layout.layout.children[0].children[0].attributes['fixed_size'] = '200px'
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cb-type-attribute.cb-sizing-fixed').attr('style') == 'flex-basis: 200px;'

    # allow wrap
    layout.layout.children[0].attributes['allow_wrap'] = True
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cb-type-stack.cb-allow_wrap')

    # titles
    layout.layout.children[0].children[0].attributes['style'] = 'title'
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cell:first-child .cb-type-attribute h3')
    layout.layout.children[0].children[0].attributes['style'] = 'subtitle'
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cell:first-child .cb-type-attribute h4')
    layout.layout.children[0].attributes['card_title_mode'] = 'auto'
    layout.layout.children[0].attributes['inner_card_title_levels'] = 'h1'
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cell:first-child h2.cell--title')
    assert resp.pyquery('.cell:first-child .cb-type-attribute h2')

    layout.layout.children[0].children[1].attributes['style'] = 'title'
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cell:first-child .cb-type-text h1')
    layout.layout.children[0].children[1].attributes['style'] = 'subtitle'
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cell:first-child .cb-type-text h2')

    # custom title
    layout.layout.children[0].attributes['card_title_mode'] = 'custom'
    layout.layout.children[0].attributes[
        'card_title_template'
    ] = 'Custom title: {{data.fields.form_display_name}}'
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cell:first-child h2.cell--title').text() == 'Custom title: Title 1'

    # unconfigured
    layout.layout.children[0].children[0].attributes['attribute'] = ''
    layout.save()
    resp = app.get(url)
    assert not resp.pyquery('.cb-type-attribute')
    layout.layout.children[0].children[0].attributes['attribute'] = 'form_display_name'  # restore

    # link text part to item
    layout.layout.children[0].children[1].attributes['link_to_item'] = True
    layout.save()
    resp = app.get(url)
    assert [x.attrib['href'] for x in resp.pyquery('.cb-type-stack a')] == [
        'http://127.0.0.1:8999/form/1/',
        'http://127.0.0.1:8999/form/2/',
        'http://127.0.0.1:8999/form/3/',
    ]
    assert resp.pyquery('.cb-type-stack a > .cb-type-text')


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_content_cell_empty_attribute(mock_send, app, admin_user):
    layout = CellLayout.objects.create(title='Layout')
    layout.data_source = CardsDataSource(
        provider_id='wcs_cards', service_id='default', slug='card_model_1', label='xxx'
    )
    layout._layout = {
        'type': 'card-root',
        'uuid': None,
        'children': [
            {
                'type': 'attribute',
                'uuid': str(uuid.uuid4()),
                'children': [],
                'attributes': {
                    'style': 'text',
                    'attribute': 'fieldg',
                    'column_label': 'Label',
                    'label_display': 'no',
                    'text_if_empty': '',
                    'empty_behaviour': 'hide',
                },
            }
        ],
    }
    layout.save()

    page = Page.objects.create(title='One', slug='one', template_name='standard')
    cell = ContentCell.objects.create(page=page, order=0, placeholder='content')
    cell._data_source = layout._data_source
    cell.attributes['layout'] = str(layout.uuid)
    cell.save()

    app = login(app)
    resp = app.get('/one/')
    url = resp.pyquery('.content-cell[data-ajax-cell-url]').attr('data-ajax-cell-url')
    resp = app.get(url)
    assert resp.pyquery('.cb-type-attribute').text() == 'test@localhost'
    layout.layout.children[0].attributes['empty_behaviour'] = 'text'
    layout.layout.children[0].attributes['text_if_empty'] = 'no data'
    layout.save()
    resp = app.get(url)
    assert resp.pyquery('.cb-type-attribute').text() == 'test@localhost no data no data'
