# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import dataclasses
import itertools

from django import forms
from django.template import RequestContext, Template, TemplateSyntaxError, VariableDoesNotExist
from django.utils.html import escape, format_html
from django.utils.safestring import mark_safe
from django.utils.translation import gettext_lazy as _
from django.utils.translation import pgettext_lazy

from combo.apps.content.utils.data_sources import DataSource, DataSourceData, DataSourceProvider
from combo.utils import requests

from .templatetags.wcs import make_public_url
from .utils import get_wcs_json, get_wcs_services


def get_concerned_user(context):
    user = context.get('selected_user') or getattr(context.get('request'), 'user', None)
    return user if user and not user.is_anonymous else None


class CardData(DataSourceData):
    def get_value(self, attribute_name, layout_block=None, context=None):
        if attribute_name == 'metadata:status':
            return self._data.get('workflow', {}).get('status', {}).get('name')
        elif attribute_name.startswith('metadata:'):
            attribute_name = attribute_name.removeprefix('metadata:')
            return self._data.get(attribute_name)

        if attribute_name in self._data['fields']:
            field_schema = self.data_source.get_field_schema(attribute_name)
            value = self._data['fields'][attribute_name]
        elif attribute_name in self._data.get('workflow', {}).get('fields', {}):
            field_schema = self.data_source.get_field_schema(attribute_name)
            value = self._data['workflow']['fields'][attribute_name]
        else:
            return ''

        if field_schema['type'] == 'text' and value:
            if field_schema.get('display_mode') in ('rich', 'basic-rich'):
                return mark_safe(value)
            if field_schema.get('display_mode') == 'pre':
                return mark_safe('<p class="plain-text-pre">%s</p>' % escape(value))
            return mark_safe(
                '<p>' + '\n'.join([(escape(x) or '</p><p>') for x in value.splitlines()]) + '</p>'
            )

        elif field_schema['type'] == 'email' and value:
            if layout_block and layout_block.attributes.get('email_display_mode', 'link') == 'link':
                return format_html('<a href="mailto:{}">{}</a>', value, value)

        elif field_schema['type'] == 'file' and value:
            if not (context and context.get('request')):
                value = value['filename']
            else:
                request = context.get('request')
                if value.get('content_type', '').startswith('image/'):
                    url = make_public_url(None, value['thumbnail_url'], request)
                    return mark_safe(f'<img src="{url}" alt="">')
                else:
                    url = make_public_url(None, value['url'], request)
                    return mark_safe(
                        f'<a class="pk-card-field-filename" href="{url}">{value["filename"]}</a>'
                    )

        return value

    def title(self):
        return self._data.get('text')

    def url(self, context=None, layout=None):
        return self._data.get('backoffice_url')


@dataclasses.dataclass
class CardsOrFormsDataSource(DataSource):
    custom_view: str = None
    selection: str = None
    ids_template: str = None

    def refresh_cache(self, force=False):
        service = get_wcs_services()[self.service_id]
        resp = get_wcs_json(service, self.get_schema_url(), cache_duration=300, invalidate_cache=force)
        self.cache = {
            'schema': resp,
        }

    def get_dependencies(self):
        wcs_site_url = get_wcs_services().get(self.service_id)['url']
        wcs_type = 'cards' if isinstance(self, CardsDataSource) else 'forms'

        urls = {
            'export': f'{wcs_site_url}api/export-import/{wcs_type}/{self.slug}/',
            'dependencies': f'{wcs_site_url}api/export-import/{wcs_type}/{self.slug}/dependencies/',
            'redirect': f'{wcs_site_url}api/export-import/{wcs_type}/{self.slug}/redirect/',
        }
        yield {'type': wcs_type, 'id': self.slug, 'text': self.get_schema().get('name'), 'urls': urls}

    def get_schema(self):
        if not self.cache:
            self.refresh_cache()
        return self.cache['schema']

    def get_field_schema(self, attribute_name):
        for field_schema in itertools.chain(
            self.get_schema().get('fields') or [],
            self.get_schema().get('workflow', {}).get('fields') or [],
        ):
            if field_schema.get('varname') == attribute_name:
                return field_schema

    def get_selection_option_fields(self, cell):
        fields = {}

        # selection
        selection_choices = [('all', self.selection_all_label)]
        with_sub_slug = any(x.sub_slug for x in cell.page.get_parents_and_self())
        if with_sub_slug:
            selection_choices.append(('url', self.selection_url_label))
        selection_choices.append(('template', _('Template')))

        fields['selection'] = forms.ChoiceField(
            label=_('Selection'),
            required=False,
            choices=selection_choices,
        )
        fields['selection'].widget.attrs['data-dynamic-display-parent'] = 'true'

        fields['ids_template'] = forms.CharField(required=False, max_length=200)
        fields['ids_template'].widget.container_attrs = {
            'data-dynamic-display-child-of': '$prefixselection',
            'data-dynamic-display-value': 'template',
        }

        return fields

    def get_form_fields_for_attribute(self, attribute_name):
        field_schema = self.get_field_schema(attribute_name)
        form_fields = {}
        if field_schema and field_schema.get('type') == 'email':
            form_fields['email_display_mode'] = forms.ChoiceField(
                label=_('Display mode'),
                required=False,
                initial='link',
                choices=[('link', _('Link')), ('text', _('Text'))],
            )
        return form_fields

    def get_info_fields(self):
        return [
            {'label': _('Identifier'), 'varname': 'metadata:id', 'type': 'string'},
            {'label': _('Receipt date'), 'varname': 'metadata:receipt_time', 'type': 'datetime'},
            {'label': _('Last modified'), 'varname': 'metadata:last_update_time', 'type': 'datetime'},
            {'label': _('Status'), 'varname': 'metadata:status', 'type': 'string'},
            {'label': _('Text'), 'varname': 'metadata:text', 'type': 'string'},
        ]

    def get_attribute_label(self, attribute_id):
        if not attribute_id:
            return '(unset)'

        if attribute_id.startswith('metadata:'):
            return [x['label'] for x in self.get_info_fields() if x['varname'] == attribute_id][0]

        for field in itertools.chain(
            self.get_schema().get('fields') or [],
            (self.get_schema().get('workflow') or {}).get('fields') or [],
        ):
            if field.get('varname') == attribute_id:
                return field.get('label')

    def get_attribute_choices(self):
        choices = [
            ('Info', [(x['varname'], x['label']) for x in self.get_info_fields()]),
            ('Data', [(x['varname'], x['label']) for x in self.get_schema()['fields'] if x.get('varname')]),
        ]
        if self.get_schema().get('workflow'):
            choices.append(
                (
                    _('Workflow Data'),
                    [
                        (x['varname'], x['label'])
                        for x in self.get_schema()['workflow']['fields']
                        if x.get('varname')
                    ],
                )
            )
        return choices

    def get_request_user_kwargs(self, context):
        return {
            'user': context.get('user'),
        }

    def get_request_parameters(self, context, paginate_by=None, page_offset=None):
        get_params = [
            'response_type=dict',
            'include-fields=on',
            'include-submission=on',
            'include-workflow=on',
            'include-actions=on',
        ]
        if self.selection == 'url':
            identifier_variable_name = f'{self.slug}_id'
            get_params += ['filter-identifier=%s' % context[identifier_variable_name]]
        elif self.selection == 'template':
            try:
                request_context = RequestContext(context['request'])
                request_context.push(context)
                ids = Template(self.ids_template).render(request_context).split(',')
                ids = [i.strip() for i in ids if i.strip()]
            except (VariableDoesNotExist, TemplateSyntaxError):
                ids = []
            get_params += ['filter-identifier=%s' % ','.join(ids)]
        if paginate_by:
            get_params += ['limit=%s' % paginate_by, 'offset=%s' % page_offset]

        return get_params

    def get_data(self, context, paginate_by=None, page_offset=None):
        from .utils import get_wcs_services

        if self.selection == 'url' and f'{self.slug}_id' not in context:
            return {'data': [], 'count': 0, 'err': 1}

        synchronous = context.get('synchronous')
        wcs_site = get_wcs_services().get(self.service_id)
        api_url = self.get_data_api_url()
        api_url += '?%s' % '&'.join(
            self.get_request_parameters(context, paginate_by=paginate_by, page_offset=page_offset)
        )

        response = requests.get(
            api_url,
            remote_service=wcs_site,
            cache_duration=5,
            raise_if_not_cached=not synchronous,
            log_errors=False,
            django_request=context.get('request'),
            **self.get_request_user_kwargs(context),
        )
        if not response.ok:
            return {'data': [], 'count': 0, 'err': 1}  # pragma: no cover

        return {
            'data': [CardData(data_source=self, data=x) for x in response.json().get('data')],
            'count': response.json().get('count'),
            'err': 0,
        }


@dataclasses.dataclass
class CardsDataSource(CardsOrFormsDataSource):
    with_user: bool = True
    linked_to_user: bool = False

    selection_all_label = _('All cards')
    selection_url_label = _('Card whose identifier is in the URL')

    def get_schema_url(self):
        return f'/api/cards/{self.slug}/@schema'

    def get_request_user_kwargs(self, context):
        return {
            'user': context.get('user') if self.with_user else None,
            'without_user': not bool(self.with_user),
        }

    def get_selection_option_fields(self, cell):
        fields = {}

        # custom view
        schema = self.get_schema()
        custom_views = schema.get('custom_views', [])
        fields['custom_view'] = forms.ChoiceField(
            label=_('Custom view'),
            required=False,
            choices=[(None, '')] + [(x['id'], x['text']) for x in custom_views],
        )
        if len(fields['custom_view'].choices) == 1:
            fields['custom_view'].widget = forms.HiddenInput()

        fields.update(super().get_selection_option_fields(cell))

        # (= !without_user from WcsCardCell)
        fields['with_user'] = forms.BooleanField(
            label=_('Restrict to cards accessible to the user'), required=False, initial=True
        )

        # (= only_for_user from WcsCardCell)
        fields['linked_to_user'] = forms.BooleanField(
            label=_('Restrict to cards linked to the logged-in user'), required=False, initial=True
        )

        return fields

    def get_data_api_url(self):
        if self.custom_view:
            api_url = f'/api/cards/{self.slug}/list/{self.custom_view}'
        else:
            api_url = f'/api/cards/{self.slug}/list'
        return api_url

    def get_request_parameters(self, context, **kwargs):
        params = super().get_request_parameters(context, **kwargs)
        if self.linked_to_user:
            user = get_concerned_user(context)
            if user:
                params += ['filter-user-uuid=%s' % user.get_name_id()]
        return params


class FormsDataSource(CardsOrFormsDataSource):
    selection_all_label = _('All forms')
    selection_url_label = _('Form whose identifier is in the URL')

    def get_schema_url(self):
        return f'/api/formdefs/{self.slug}/schema'

    def get_data_api_url(self):
        if self.custom_view:
            api_url = f'/api/forms/{self.slug}/list/{self.custom_view}'
        else:
            api_url = f'/api/forms/{self.slug}/list'
        return api_url


class CardsOrFormsDataSourceProvider(DataSourceProvider):
    @property
    def choices(self):
        from .utils import get_wcs_json, get_wcs_services

        wcs_services = get_wcs_services()
        card_data_source_items = []

        for key, service in wcs_services.items():
            resp = get_wcs_json(service, self.wcs_base_url)
            if resp and resp.get('data'):
                card_data_source_items.extend(
                    self.data_source_class(
                        provider_id=self.provider_id,
                        service_id=key,
                        slug=x['slug'],
                        label=x['text'] if len(wcs_services) == 1 else f'{service["title"]}: {x["text"]}',
                    )
                    for x in resp.get('data')
                )
        return card_data_source_items


class CardsDataSourceProvider(CardsOrFormsDataSourceProvider):
    provider_id = 'wcs_cards'
    title = pgettext_lazy('data-source', 'Cards')
    data_source_class = CardsDataSource
    wcs_base_url = 'api/cards/@list'


class FormsDataSourceProvider(CardsOrFormsDataSourceProvider):
    provider_id = 'wcs_forms'
    title = pgettext_lazy('data-source', 'Forms')
    data_source_class = FormsDataSource
    wcs_base_url = 'api/forms/@list'


class UserFormData(DataSourceData):
    def get_value(self, attribute_name, layout_block=None, context=None):
        if attribute_name == 'metadata:text':
            return self.title()
        return self._data.get(attribute_name)

    def title(self):
        return self._data.get('form_display_name')

    def url(self, context=None, layout=None):
        return self._data.get('form_url')


@dataclasses.dataclass
class UserFormsDataSource(DataSource):
    include_open_forms: bool = True
    include_closed_forms: bool = False
    include_drafts: bool = False
    include_forms_user_can_access: bool = False

    def get_attribute_choices(self):
        return [
            ('form_display_name', _('Full label')),
            ('form_name', _('Form name')),
            ('form_number', _('Number')),
            ('form_receipt_date', _('Receipt date')),
            ('form_receipt_time', _('Receipt time')),
            ('form_status', _('Status')),
        ]

    def get_selection_option_fields(self, cell):
        fields = {}

        fields['include_open_forms'] = forms.BooleanField(
            label=_('Include open forms'), required=False, initial=self.include_open_forms
        )
        fields['include_closed_forms'] = forms.BooleanField(
            label=_('Include done forms'), required=False, initial=self.include_closed_forms
        )
        fields['include_drafts'] = forms.BooleanField(
            label=_('Include drafts'), required=False, initial=self.include_drafts
        )
        fields['include_forms_user_can_access'] = forms.BooleanField(
            label=_('Include forms to which the user can access'),
            required=False,
            initial=self.include_forms_user_can_access,
        )
        return fields

    def get_data(self, context, paginate_by=None, page_offset=None):
        from .utils import get_wcs_services

        synchronous = context.get('synchronous')
        wcs_site = get_wcs_services().get(self.service_id)

        url = '/api/user/forms/'
        if not (self.include_open_forms or self.include_closed_forms):
            url = '/api/user/drafts'
        user = get_concerned_user(context)
        if user:
            user_name_id = user.get_name_id()
            if user_name_id:
                url = '/api/users/%s/forms' % user_name_id
                if not (self.include_open_forms or self.include_closed_forms):
                    url = '/api/users/%s/drafts' % user_name_id

        params = []
        if self.include_open_forms and self.include_closed_forms:
            params.append('status=all')
        elif self.include_closed_forms:
            params.append('status=done')
        else:
            params.append('status=open')
        if self.include_forms_user_can_access:
            params.append('include-accessible=on')
        if self.include_drafts:
            params.append('include-drafts=on')

        params.append('sort=desc')

        url += '?' + '&'.join(params)

        response = requests.get(
            url,
            remote_service=wcs_site,
            cache_duration=5,
            raise_if_not_cached=not synchronous,
            log_errors=False,
            django_request=context.get('request'),
            user=user,
        )
        if not response.ok:
            return {'data': [], 'count': 0, 'err': 1}  # pragma: no cover

        count = len(response.json()['data'])
        if paginate_by:
            # pagination is not supported server-side
            data = response.json()['data'][page_offset : page_offset + paginate_by]
        else:
            data = response.json()['data']
        return {
            'data': [UserFormData(data_source=self, data=x) for x in data],
            'count': count,
            'err': 0,
        }


class UserFormsDataSourceProvider(DataSourceProvider):
    provider_id = 'wcs_user_forms'
    title = pgettext_lazy('data-source', 'User forms')
    data_source_class = UserFormsDataSource

    @property
    def choices(self):
        from .utils import get_wcs_services

        wcs_services = get_wcs_services()
        data_source_items = []

        for key, service in wcs_services.items():
            data_source_items.append(
                UserFormsDataSource(
                    provider_id=self.provider_id,
                    service_id=key,
                    slug='__user_forms',
                    label=(
                        str(_('User Forms'))
                        if len(wcs_services) == 1
                        else f'{service["title"]}: {_("User Forms")}'
                    ),
                )
            )
        return data_source_items
