# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import uuid

from django.shortcuts import get_object_or_404
from django.urls import reverse, reverse_lazy
from django.views.generic import (
    CreateView,
    DeleteView,
    DetailView,
    FormView,
    ListView,
    RedirectView,
    UpdateView,
)

from combo.data.models import Page
from combo.utils import is_ajax

from .forms import LayoutAddBlockForm, LayoutAddForm, LayoutDeleteBlockForm
from .models import CellLayout
from .utils.layouts import create_block_from_json_data, get_block_class_by_type


class HomeView(ListView):
    model = CellLayout
    template_name = 'combo/layouts/manager_home.html'


class LayoutAddView(CreateView):
    template_name = 'combo/layouts/manager_layout_form.html'
    model = CellLayout
    form_class = LayoutAddForm

    def form_valid(self, form):
        response = super().form_valid(form)
        return response

    def get_success_url(self):
        return reverse('manager-layouts-view', kwargs={'slug': self.object.slug})


class LayoutDetailView(DetailView):
    template_name = 'combo/layouts/manager_layout_view.html'
    model = CellLayout

    def get_template_names(self):
        if is_ajax(self.request):
            return ['combo/layouts/manager_layout_view_layout_fragment.html']
        return ['combo/layouts/manager_layout_view.html']

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        if not is_ajax(self.request):
            context['pages'] = Page.objects.filter(
                univercell__attributes__layout=str(self.object.uuid)
            ).order_by('title')
        if f'layout-{self.object.id}-clipboard' in self.request.session:
            context['has_clipboard_content'] = True
        return context


class LayoutRenameView(UpdateView):
    template_name = 'combo/layouts/manager_layout_form.html'
    model = CellLayout
    fields = ['title']

    def get_success_url(self):
        return reverse('manager-layouts-view', kwargs={'slug': self.object.slug})


class LayoutRedirectByUuidView(RedirectView):
    def get_redirect_url(self, uuid):
        layout = get_object_or_404(CellLayout, uuid=uuid)
        return reverse('manager-layouts-view', kwargs={'slug': layout.slug})


class LayoutDeleteView(DeleteView):
    template_name = 'combo/layouts/manager_layout_confirm_delete.html'
    model = CellLayout
    success_url = reverse_lazy('manager-layouts-index')


class LayoutAddBlockView(FormView):
    template_name = 'combo/layouts/manager_layout_add_block.html'
    form_class = LayoutAddBlockForm

    def get_initial(self):
        initial = super().get_initial()
        initial['uuid'] = self.request.GET['active-block-uuid']
        return initial

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['layout_slug'] = self.kwargs['slug']
        return context

    def form_valid(self, form):
        layout = CellLayout.objects.get(slug=self.kwargs['slug'])
        parent_uuid = form.cleaned_data['uuid']
        self.new_block = get_block_class_by_type(type=form.cleaned_data['type'])(
            type=form.cleaned_data['type'], uuid=str(uuid.uuid4())
        )
        layout.layout.add_block(self.new_block, root_uuid=parent_uuid)
        layout.save()
        return super().form_valid(form)

    def get_success_url(self):
        params = f'?active-block-uuid={self.new_block.uuid}'
        if not self.new_block.is_container:
            params += '&open-properties=true'
        return reverse('manager-layouts-view', kwargs=self.kwargs) + params


class LayoutDeleteBlockView(FormView):
    template_name = 'combo/layouts/manager_layout_delete_block.html'
    form_class = LayoutDeleteBlockForm

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['layout_slug'] = self.kwargs['slug']
        return context

    def get_initial(self):
        initial = super().get_initial()
        initial['uuid'] = self.request.GET['active-block-uuid']
        return initial

    def form_valid(self, form):
        layout = CellLayout.objects.get(slug=self.kwargs['slug'])
        block_uuid = form.cleaned_data['uuid']
        layout.layout.delete_block(uuid=block_uuid)
        layout.save()
        return super().form_valid(form)

    def get_success_url(self):
        return reverse('manager-layouts-view', kwargs=self.kwargs)


class LayoutCutBlockView(RedirectView):
    def get_redirect_url(self, **kwargs):
        layout = CellLayout.objects.get(slug=kwargs['slug'])
        block_uuid = self.request.GET['active-block-uuid']
        parent_block = layout.layout.get_parent_by_uuid(block_uuid)
        block = layout.layout.get_block_by_uuid(block_uuid)
        self.request.session[f'layout-{layout.id}-clipboard'] = block.as_json()
        layout.layout.delete_block(uuid=block_uuid)
        layout.save()
        return reverse('manager-layouts-view', kwargs=kwargs) + f'?active-block-uuid={parent_block.uuid}'


class LayoutCopyBlockView(RedirectView):
    def get_redirect_url(self, **kwargs):
        layout = CellLayout.objects.get(slug=kwargs['slug'])
        block_uuid = self.request.GET['active-block-uuid']
        block = layout.layout.get_block_by_uuid(block_uuid)
        self.request.session[f'layout-{layout.id}-clipboard'] = block.as_json()
        return reverse('manager-layouts-view', kwargs=kwargs) + f'?active-block-uuid={block_uuid}'


class LayoutPasteBlockView(RedirectView):
    def get_redirect_url(self, **kwargs):
        layout = CellLayout.objects.get(slug=kwargs['slug'])
        block_uuid = self.request.GET['active-block-uuid']
        parent_block = layout.layout.get_block_by_uuid(block_uuid)
        clipboard_block = create_block_from_json_data(self.request.session[f'layout-{layout.id}-clipboard'])

        def collect_uuids(blocks):
            for block in blocks:
                yield block.uuid
                yield from collect_uuids(block.children)

        existing_uuids = set(collect_uuids([layout.layout]))

        # do not create duplicated UUIDs
        def replace_uuids(blocks):
            for block in blocks:
                if block.uuid in existing_uuids:
                    block.uuid = str(uuid.uuid4())
                if block.children:
                    replace_uuids(block.children)

        replace_uuids([clipboard_block])
        parent_block.children.append(clipboard_block)
        layout.save()
        return reverse('manager-layouts-view', kwargs=kwargs) + f'?active-block-uuid={parent_block.uuid}'


class LayoutMoveBlockBeforeView(RedirectView):
    def get_redirect_url(self, **kwargs):
        layout = CellLayout.objects.get(slug=kwargs['slug'])
        block_uuid = self.request.GET['active-block-uuid']
        parent_block = layout.layout.get_parent_by_uuid(block_uuid)
        position = [x.uuid for x in parent_block.children].index(block_uuid)
        if position > 0:
            block = parent_block.children.pop(position)
            parent_block.children[position - 1 : position - 1] = [block]
            layout.save()
        return reverse('manager-layouts-view', kwargs=kwargs) + f'?active-block-uuid={block_uuid}'


class LayoutMoveBlockAfterView(RedirectView):
    def get_redirect_url(self, **kwargs):
        layout = CellLayout.objects.get(slug=kwargs['slug'])
        block_uuid = self.request.GET['active-block-uuid']
        parent_block = layout.layout.get_parent_by_uuid(block_uuid)
        position = [x.uuid for x in parent_block.children].index(block_uuid)
        if position < (len(parent_block.children) - 1):
            block = parent_block.children.pop(position)
            parent_block.children[position + 1 : position + 1] = [block]
            layout.save()

        return reverse('manager-layouts-view', kwargs=kwargs) + f'?active-block-uuid={block_uuid}'


class LayoutBlockOptionsView(FormView):
    template_name = 'combo/layouts/manager_layout_block_options.html'

    def dispatch(self, request, *args, **kwargs):
        self.layout = CellLayout.objects.get(slug=kwargs['slug'])
        self.layout_block_uuid = request.GET['active-block-uuid']
        self.layout_block = self.layout.layout.get_block_by_uuid(uuid=self.layout_block_uuid)
        return super().dispatch(request, *args, **kwargs)

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['layout_slug'] = self.kwargs['slug']
        return context

    def get_form_class(self):
        return self.layout_block.get_options_form_class()

    def get_form_kwargs(self):
        kwargs = super().get_form_kwargs()
        kwargs['data_source'] = self.layout.data_source
        kwargs['layout_block'] = self.layout_block
        return kwargs

    def get_initial(self):
        initial = super().get_initial()
        initial['uuid'] = self.request.GET['active-block-uuid']
        initial.update({x: getattr(self.layout_block, x) for x in ('type',)})
        initial.update(self.layout_block.attributes or {})
        return initial

    def form_valid(self, form):
        if 'type' in form.cleaned_data:
            self.layout_block.type = form.cleaned_data.pop('type')
        form.cleaned_data.pop('uuid')
        self.layout_block.attributes.update(form.cleaned_data)
        self.layout.save()
        return super().form_valid(form)

    def get_success_url(self):
        return (
            reverse('manager-layouts-view', kwargs=self.kwargs)
            + f'?active-block-uuid={self.layout_block_uuid}'
        )
