import urllib
from unittest import mock

import pytest
from django.utils.module_loading import import_string

from combo.apps.content.models import CellLayout, ContentCell
from combo.apps.content.utils import layouts
from combo.apps.wcs.data_sources import UserFormsDataSource
from combo.data.models import Page

from .test_manager import login
from .utils import manager_submit_cell
from .wcs.utils import mocked_requests_send

pytestmark = pytest.mark.django_db


def test_content_cell(app, admin_user):
    page = Page.objects.create(title='One', slug='one', template_name='standard')
    app = login(app)
    resp = app.get('/manage/pages/%s/' % page.pk)
    data_add_url = [x for x in resp.html.find_all('option') if x.text == 'Content'][0].get('data-add-url')
    resp = app.get(data_add_url).follow()
    cell = ContentCell.objects.all().first()
    base_fields = {
        x.removeprefix(f'c{cell.get_reference()}-')
        for x in resp.form.fields
        if x and x.startswith(f'c{cell.get_reference()}')
    }
    assert [x[0] for x in resp.form[f'c{cell.get_reference()}-_data_source'].options] == [
        'wcs_user_forms:default:__user_forms',
        'wcs_user_forms:other:__user_forms',
    ]
    resp.form[f'c{cell.get_reference()}-_data_source'] = 'wcs_user_forms:default:__user_forms'
    manager_submit_cell(resp.forms[0], update_form_mode=True)
    # data source is selected, specific parameters are displayed
    assert {
        x.removeprefix(f'c{cell.get_reference()}-')
        for x in resp.form.fields
        if x and x.startswith(f'c{cell.get_reference()}')
    } - base_fields == {
        'include_open_forms',
        'include_closed_forms',
        'include_drafts',
        'include_forms_user_can_access',
    }
    assert resp.form[f'c{cell.get_reference()}-include_open_forms'].checked  # default value
    assert not resp.form[f'c{cell.get_reference()}-include_drafts'].checked  # default value


def test_cell_layout(app, admin_user):
    app = login(app)
    resp = app.get('/manage/layouts/')
    resp = resp.click('New layout')
    resp.form['title'] = 'test layout'
    resp.form['_data_source'] = 'wcs_user_forms:default:__user_forms'
    resp.form['mode'] = 'card'
    resp = resp.form.submit('submit').follow()
    assert [x.slug for x in CellLayout.objects.all()] == ['test-layout']

    resp = resp.click('Rename')
    resp.form['title'] = 'Renamed test layout'
    resp = resp.form.submit('submit').follow()
    assert [x.title for x in CellLayout.objects.all()] == ['Renamed test layout']

    resp = resp.click('Delete')
    assert 'Are you sure you want to delete this layout?' in resp.text
    resp = resp.form.submit('submit').follow()
    assert not CellLayout.objects.all().exists()

    resp = resp.click('New layout')
    resp.form['title'] = 'test layout'
    resp.form['_data_source'] = 'wcs_user_forms:default:__user_forms'
    resp.form['mode'] = 'card'
    resp = resp.form.submit('submit').follow()

    layout = CellLayout.objects.all().first()

    stack_uuid = resp.pyquery('layout-block').attr('uuid')
    assert stack_uuid == layout.layout.children[0].uuid
    add_url = resp.pyquery('#toolbar-add').attr('data-base-url') + f'?active-block-uuid={stack_uuid}'
    resp = app.get(add_url)
    resp.form['type'] = 'row'
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    assert layout.layout.children[0].children[0].type

    resp = app.get(add_url)
    resp.form['type'] = 'attribute'
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    assert [x.type for x in layout.layout.children[0].children] == ['row', 'attribute']

    row_uuid = layout.layout.children[0].children[0].uuid
    move_after_url = (
        resp.pyquery('#toolbar-move-after').attr('data-base-url') + f'?active-block-uuid={row_uuid}'
    )
    app.get(move_after_url)
    layout.refresh_from_db()
    assert [x.type for x in layout.layout.children[0].children] == ['attribute', 'row']
    app.get(move_after_url)  # again but last, no change
    layout.refresh_from_db()
    assert [x.type for x in layout.layout.children[0].children] == ['attribute', 'row']

    # again, simulating ajax call, to check for content
    ajax_resp = app.get(move_after_url, headers={'x-requested-with': 'XMLHttpRequest'}).follow(
        headers={'x-requested-with': 'XMLHttpRequest'}
    )
    assert ajax_resp.text.startswith('<div id="layout"')

    move_before_url = (
        resp.pyquery('#toolbar-move-before').attr('data-base-url') + f'?active-block-uuid={row_uuid}'
    )
    app.get(move_before_url)
    layout.refresh_from_db()
    assert [x.type for x in layout.layout.children[0].children] == ['row', 'attribute']

    delete_url = resp.pyquery('#toolbar-delete').attr('data-base-url') + f'?active-block-uuid={row_uuid}'
    resp = app.get(delete_url)
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    assert [x.type for x in layout.layout.children[0].children] == ['attribute']

    resp = app.get(add_url)
    resp.form['type'] = 'row'
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    assert layout.layout.children[0].children[-1].type == 'row'

    row_uuid = layout.layout.children[0].children[-1].uuid
    options_url = resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={row_uuid}'
    resp = app.get(options_url)
    assert resp.form['sizing'].value == 'fit'
    resp.form['sizing'] = 'grow'
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    assert layout.layout.children[0].children[-1].attributes.get('sizing') == 'grow'

    resp = app.get(options_url)
    assert resp.form['sizing'].value == 'grow'
    resp = resp.form.submit('submit').follow()

    options_url = resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={row_uuid}'
    resp = app.get(options_url)
    resp = resp.form.submit('submit').follow()

    options_url = resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={stack_uuid}'
    resp = app.get(options_url)
    assert set(resp.form.fields.keys()).issuperset({'card_title_mode', 'card_title_template'})

    # change type
    resp.form['type'] = 'row'
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    assert layout.layout.children[0].type == 'row'

    # change type
    resp = app.get(options_url)
    resp.form['type'] = 'group'
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    assert layout.layout.children[0].type == 'group'


def test_cell_layout_card_title(app, admin_user):
    app = login(app)
    resp = app.get('/manage/layouts/')
    resp = resp.click('New layout')
    resp.form['title'] = 'test layout'
    resp.form['_data_source'] = 'wcs_user_forms:default:__user_forms'
    resp.form['mode'] = 'card'
    resp = resp.form.submit('submit').follow()

    layout = CellLayout.objects.all().first()
    row_uuid = layout.layout.children[0].uuid
    options_url = resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={row_uuid}'
    resp = app.get(options_url)
    assert resp.form['card_title_mode'].value == 'none'
    resp.form['card_title_mode'] = 'auto'
    resp = resp.form.submit('submit').follow()
    assert resp.pyquery('.content-before-layout').text() == 'Automatic title from content'

    resp = app.get(options_url)
    resp.form['card_title_mode'] = 'custom'
    resp.form['card_title_template'] = 'hello world'
    resp = resp.form.submit('submit').follow()
    assert resp.pyquery('.content-before-layout').text() == 'Custom title from template'


def test_cell_layout_unique_slugs(app):
    CellLayout.objects.create(title='custom layout')
    layout2 = CellLayout.objects.create(title='custom layout')
    layout3 = CellLayout.objects.create(title='custom layout')
    assert layout2.slug == 'custom-layout-2'
    assert layout3.slug == 'custom-layout-3'


def test_cell_layout_table(app, admin_user):
    app = login(app)
    resp = app.get('/manage/layouts/')
    resp = resp.click('New layout')
    resp.form['title'] = 'test layout'
    resp.form['_data_source'] = 'wcs_user_forms:default:__user_forms'
    resp.form['mode'] = 'table'
    resp = resp.form.submit('submit').follow()
    assert [x.slug for x in CellLayout.objects.all()] == ['test-layout']
    layout = CellLayout.objects.all().first()
    assert layout.layout.type == 'table-root'

    columns_uuid = resp.pyquery('layout-block[type="table-columns"]').attr('uuid')
    assert columns_uuid == layout.layout.children[0].uuid
    add_url = resp.pyquery('#toolbar-add').attr('data-base-url') + f'?active-block-uuid={columns_uuid}'
    resp = app.get(add_url)
    resp.form['type'] = 'attribute'
    resp = resp.form.submit('submit').follow()
    attribute_uuid = urllib.parse.parse_qs(urllib.parse.urlparse(resp.request.url).query)[
        'active-block-uuid'
    ][0]
    assert urllib.parse.parse_qs(urllib.parse.urlparse(resp.request.url).query)['open-properties']
    options_url = (
        resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={attribute_uuid}'
    )

    resp = app.get(options_url)
    resp.form['attribute'] = 'form_display_name'
    resp.form['column_label'] = 'column'
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    assert layout.layout.children[0].children[0].attributes['column_label'] == 'column'
    assert not resp.pyquery('layout-block[type="table-columns"]').attr('options')


def test_cell_layout_uuid_redirect(app, admin_user):
    layout = CellLayout.objects.create(title='Test')
    app = login(app)
    resp = app.get(f'/manage/layouts/layout/by-uuid/{layout.uuid}/')
    assert resp.location == '/manage/layouts/layout/test/'


def test_content_cell_layout(app, admin_user):
    layout = CellLayout.objects.create(title='Layout')
    layout.data_source = UserFormsDataSource(
        provider_id='wcs_user_forms', service_id='default', slug='__user_forms', label='test: User Forms'
    )
    layout.save()
    page = Page.objects.create(title='One', slug='one', template_name='standard')
    cell = ContentCell.objects.create(page=page, order=0, placeholder='content')
    app = login(app)
    resp = app.get('/manage/pages/%s/' % page.pk)
    resp.form[f'c{cell.get_reference()}-_data_source'] = 'wcs_user_forms:default:__user_forms'
    manager_submit_cell(resp.forms[0])
    resp = app.get('/manage/pages/%s/' % page.pk)
    assert [x[2] for x in resp.form[f'c{cell.get_reference()}-layout'].options] == ['Simple list', 'Layout']
    resp.form[f'c{cell.get_reference()}-layout'] = str(layout.uuid)
    manager_submit_cell(resp.forms[0])
    cell.refresh_from_db()
    assert cell.attributes['layout'] == str(layout.uuid)


def test_cell_layout_attribute(app, admin_user):
    app = login(app)
    resp = app.get('/manage/layouts/')
    resp = resp.click('New layout')
    resp.form['title'] = 'test layout'
    resp.form['_data_source'] = 'wcs_user_forms:default:__user_forms'
    resp.form['mode'] = 'card'
    resp = resp.form.submit('submit').follow()

    cell = CellLayout.objects.all().first()

    stack_uuid = resp.pyquery('layout-block').attr('uuid')
    assert stack_uuid == cell.layout.children[0].uuid
    add_url = resp.pyquery('#toolbar-add').attr('data-base-url') + f'?active-block-uuid={stack_uuid}'
    resp = app.get(add_url)
    resp.form['type'] = 'attribute'
    resp = resp.form.submit('submit').follow()

    cell.refresh_from_db()
    assert [x.type for x in cell.layout.children[0].children] == ['attribute']

    attribute_uuid = cell.layout.children[0].children[0].uuid

    options_url = (
        resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={attribute_uuid}'
    )
    resp = app.get(options_url)
    assert [x[0] for x in resp.form['attribute'].options] == [
        'form_display_name',
        'form_name',
        'form_number',
        'form_receipt_date',
        'form_receipt_time',
        'form_status',
    ]
    resp.form['attribute'] = 'form_receipt_date'
    resp.form['label_display'] = 'above'
    resp = resp.form.submit('submit').follow()
    assert resp.pyquery('layout-block[type="attribute"]').text() == 'Receipt date\nReceipt date'

    resp = app.get(options_url)
    resp.form['label_display'] = 'before'
    resp = resp.form.submit('submit').follow()
    assert resp.pyquery('layout-block[type="attribute"]').text() == 'Receipt date: Receipt date'

    resp = app.get(options_url)
    resp.form['label_display'] = 'no'
    resp = resp.form.submit('submit').follow()
    assert resp.pyquery('layout-block[type="attribute"]').text() == 'Receipt date'


def test_cell_layout_text(app, admin_user):
    app = login(app)
    resp = app.get('/manage/layouts/')
    resp = resp.click('New layout')
    resp.form['title'] = 'test layout'
    resp.form['_data_source'] = 'wcs_user_forms:default:__user_forms'
    resp.form['mode'] = 'card'
    resp = resp.form.submit('submit').follow()

    cell = CellLayout.objects.all().first()

    stack_uuid = resp.pyquery('layout-block').attr('uuid')
    assert stack_uuid == cell.layout.children[0].uuid
    add_url = resp.pyquery('#toolbar-add').attr('data-base-url') + f'?active-block-uuid={stack_uuid}'
    resp = app.get(add_url)
    resp.form['type'] = 'text'
    resp = resp.form.submit('submit').follow()

    cell.refresh_from_db()
    assert [x.type for x in cell.layout.children[0].children] == ['text']

    text_uuid = cell.layout.children[0].children[0].uuid

    options_url = resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={text_uuid}'
    resp = app.get(options_url)
    resp.form['text'] = 'hello world'
    resp.form['style'] = 'title'
    resp = resp.form.submit('submit').follow()
    assert resp.pyquery('layout-block[type="text"]').text() == 'hello world'


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_content_cell_cards_datasource(mock_send, app, admin_user):
    page = Page.objects.create(title='One', slug='one', template_name='standard')
    app = login(app)
    resp = app.get('/manage/pages/%s/' % page.pk)
    data_add_url = [x for x in resp.html.find_all('option') if x.text == 'Content'][0].get('data-add-url')
    resp = app.get(data_add_url).follow()
    cell = ContentCell.objects.all().first()
    base_fields = {
        x.removeprefix(f'c{cell.get_reference()}-')
        for x in resp.form.fields
        if x and x.startswith(f'c{cell.get_reference()}')
    }
    resp.form[f'c{cell.get_reference()}-_data_source'] = 'wcs_cards:default:card_model_1'
    manager_submit_cell(resp.forms[0])

    cell.refresh_from_db()
    assert [(x['type'], x['id']) for x in cell.get_dependencies()] == [('cards', 'card_model_1')]

    # data source is selected, specific parameters are displayed
    assert {
        x.removeprefix(f'c{cell.get_reference()}-')
        for x in resp.form.fields
        if x and x.startswith(f'c{cell.get_reference()}')
    } - base_fields == {
        'linked_to_user',
        'ids_template',
        'with_user',
        'custom_view',
        'selection',
    }
    # no custom view -> the field is hidden
    assert resp.form[f'c{cell.get_reference()}-custom_view'].attrs == {'type': 'hidden'}
    assert [x[0] for x in resp.form[f'c{cell.get_reference()}-selection'].options] == ['all', 'template']

    page.refresh_from_db()
    page.sub_slug = 'card_model_1_id'
    page.save()
    resp = app.get('/manage/pages/%s/' % page.pk)
    assert [x[0] for x in resp.form[f'c{cell.get_reference()}-selection'].options] == [
        'all',
        'url',
        'template',
    ]


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_content_cell_forms_datasource(mock_send, app, admin_user):
    page = Page.objects.create(title='One', slug='one', template_name='standard')
    app = login(app)
    resp = app.get('/manage/pages/%s/' % page.pk)
    data_add_url = [x for x in resp.html.find_all('option') if x.text == 'Content'][0].get('data-add-url')
    resp = app.get(data_add_url).follow()
    cell = ContentCell.objects.all().first()
    base_fields = {
        x.removeprefix(f'c{cell.get_reference()}-')
        for x in resp.form.fields
        if x and x.startswith(f'c{cell.get_reference()}')
    }
    resp.form[f'c{cell.get_reference()}-_data_source'] = 'wcs_forms:default:a-private-form'
    manager_submit_cell(resp.forms[0])

    cell.refresh_from_db()
    assert [(x['type'], x['id']) for x in cell.get_dependencies()] == [('forms', 'a-private-form')]

    # data source is selected, specific parameters are displayed
    assert {
        x.removeprefix(f'c{cell.get_reference()}-')
        for x in resp.form.fields
        if x and x.startswith(f'c{cell.get_reference()}')
    } - base_fields == {
        'ids_template',
        'selection',
    }


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_cell_layout_attribute_cards_email(mock_send, app, admin_user):
    app = login(app)
    resp = app.get('/manage/layouts/')
    resp = resp.click('New layout')
    resp.form['title'] = 'test layout'
    resp.form['_data_source'] = 'wcs_cards:default:card_model_1'
    resp.form['mode'] = 'card'
    resp = resp.form.submit('submit').follow()

    cell = CellLayout.objects.all().first()

    stack_uuid = resp.pyquery('layout-block').attr('uuid')
    add_url = resp.pyquery('#toolbar-add').attr('data-base-url') + f'?active-block-uuid={stack_uuid}'
    resp = app.get(add_url)
    resp.form['type'] = 'attribute'
    resp = resp.form.submit('submit').follow()

    cell.refresh_from_db()
    assert [x.type for x in cell.layout.children[0].children] == ['attribute']

    attribute_uuid = cell.layout.children[0].children[0].uuid

    options_url = (
        resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={attribute_uuid}'
    )
    resp = app.get(options_url)
    resp.form['attribute'] = 'fieldg'
    resp.form['label_display'] = 'above'
    resp = resp.form.submit('submit').follow()
    assert resp.pyquery('layout-block[type="attribute"]').text() == 'Field G\nField G'

    # special option for email fields
    resp = app.get(options_url)
    assert [x[0] for x in resp.form['email_display_mode'].options] == ['link', 'text']


def test_layout_blocks_methods():
    for klass_name in dir(layouts):
        if not klass_name.startswith('Layout'):
            continue
        instance = import_string(f'combo.apps.content.utils.layouts.{klass_name}')()
        instance.get_options_form_class()
        instance.get_fields_for_child_form()
        if 'Root' in klass_name:
            instance.get_base_title_level()
            assert instance.get_block_by_uuid('root') is instance


def test_cell_layout_natural_key():
    layout = CellLayout.objects.create(title='custom layout')

    assert CellLayout.objects.get_by_natural_key(*layout.natural_key()).uuid == layout.uuid


def test_cell_layout_clipboard(app, admin_user):
    app = login(app)
    resp = app.get('/manage/layouts/')
    resp = resp.click('New layout')
    resp.form['title'] = 'test layout'
    resp.form['_data_source'] = 'wcs_user_forms:default:__user_forms'
    resp.form['mode'] = 'card'
    resp = resp.form.submit('submit').follow()
    assert not resp.pyquery('#layout').attr('data-has-clipboard-content')

    layout = CellLayout.objects.all().first()

    stack_uuid = resp.pyquery('layout-block').attr('uuid')

    add_url = resp.pyquery('#toolbar-add').attr('data-base-url') + f'?active-block-uuid={stack_uuid}'
    resp = app.get(add_url)
    resp.form['type'] = 'row'
    resp = resp.form.submit('submit').follow()
    layout.refresh_from_db()
    row_uuid = layout.layout.children[0].children[0].uuid

    add_url = resp.pyquery('#toolbar-add').attr('data-base-url') + f'?active-block-uuid={row_uuid}'
    resp = app.get(add_url)
    resp.form['type'] = 'text'
    resp = resp.form.submit('submit').follow()
    resp = app.get(add_url)
    resp.form['type'] = 'text'
    resp = resp.form.submit('submit').follow()

    layout.refresh_from_db()
    text_uuid = layout.layout.children[0].children[0].children[0].uuid
    options_url = resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={text_uuid}'
    resp = app.get(options_url)
    resp.form['text'] = 'hello'
    resp = resp.form.submit('submit').follow()

    layout.refresh_from_db()
    text_uuid = layout.layout.children[0].children[0].children[1].uuid
    options_url = resp.pyquery('#toolbar-options').attr('data-base-url') + f'?active-block-uuid={text_uuid}'
    resp = app.get(options_url)
    resp.form['text'] = 'world'
    resp = resp.form.submit('submit').follow()

    layout.refresh_from_db()

    cut_url = resp.pyquery('#toolbar-cut').attr('data-base-url') + f'?active-block-uuid={row_uuid}'
    resp = app.get(cut_url).follow()
    assert resp.pyquery('#layout').attr('data-has-clipboard-content')
    layout.refresh_from_db()
    assert not layout.layout.children[0].children

    paste_url = resp.pyquery('#toolbar-paste').attr('data-base-url') + f'?active-block-uuid={stack_uuid}'
    resp = app.get(paste_url).follow()
    layout.refresh_from_db()
    assert len(layout.layout.children[0].children) == 1
    assert layout.layout.children[0].children[0].uuid == row_uuid  # uuid was kept

    # paste again
    resp = app.get(paste_url).follow()
    layout.refresh_from_db()
    assert len(layout.layout.children[0].children) == 2
    assert layout.layout.children[0].children[1].uuid != row_uuid  # uuid was changed

    copy_url = resp.pyquery('#toolbar-copy').attr('data-base-url') + f'?active-block-uuid={text_uuid}'
    resp = app.get(copy_url).follow()
    resp = app.get(paste_url).follow()
    layout.refresh_from_db()
    assert len(layout.layout.children[0].children[0].children) == 2  # row kept its two text children
    assert len(layout.layout.children[0].children) == 3  # text block was copied to stack
    assert layout.layout.children[0].children[-1].type == 'text'
