# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import dataclasses
import json
import uuid

from django.core import serializers
from django.db import models
from django.utils.text import slugify
from django.utils.timezone import now
from django.utils.translation import gettext_lazy as _

from ..utils.data_sources import WithDataSourceMixin
from ..utils.layouts import get_block_class_by_type


class CellLayoutManager(models.Manager):
    def get_by_natural_key(self, uuid):
        return self.get(uuid=uuid)


class CellLayout(models.Model, WithDataSourceMixin):
    objects = CellLayoutManager()
    application_component_type = 'cell-layouts'

    uuid = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    title = models.CharField(_('Title'), max_length=150)
    slug = models.SlugField(_('Slug'), max_length=150)

    _data_source = models.JSONField(_('Data source'), blank=True, default=dict)
    _layout = models.JSONField(blank=True, default=dict)

    creation_timestamp = models.DateTimeField(default=now)
    last_update_timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = _('Cell Layout')
        ordering = ['title']

    def __str__(self):
        return self.title or super().__str__()

    def natural_key(self):
        return (str(self.uuid),)

    def get_dependencies(self):
        # no use in returning dependencies from data source as it will come with
        # the cell using this layout.
        return []

    @classmethod
    def export_all_for_json(cls):
        return [x.get_as_serialized_object() for x in cls.objects.all()]

    def get_as_serialized_object(self):
        serialized = json.loads(
            serializers.serialize(
                'json', [self], use_natural_foreign_keys=True, use_natural_primary_keys=True
            )
        )[0]
        del serialized['model']
        serialized['fields'].pop('creation_timestamp')
        serialized['fields'].pop('last_update_timestamp')
        serialized['fields']['_data_source'].pop('cache', None)
        return serialized

    @classmethod
    def load_serialized_objects(cls, json_site):
        for json_layout in json_site:
            cls.load_serialized_object(json_layout)

    @classmethod
    def load_serialized_object(cls, json_layout):
        json_layout['model'] = 'content.celllayout'
        layout, dummy = cls.objects.get_or_create(uuid=json_layout['fields']['uuid'])
        json_layout['pk'] = layout.id
        layout = next(serializers.deserialize('json', json.dumps([json_layout]), ignorenonexistent=True))
        if not layout.object.last_update_timestamp:
            layout.object.last_update_timestamp = now()
        layout.save()

    def refresh_from_db(self):
        super().refresh_from_db()
        self._root_layout_object = None
        self._data_source_object = None

    def save(self, *args, **kwargs):
        if self._root_layout_object:
            self._layout = self._root_layout_object.as_json()
        if self._data_source:
            self.data_source.refresh_cache()
            self._data_source = dataclasses.asdict(self.data_source)
        if not self.slug:
            base_slug = slugify(self.title)[:45]
            slug = base_slug
            i = 2
            while CellLayout.objects.filter(slug=slug).exists():
                slug = '%s-%s' % (base_slug, i)
                i += 1
            self.slug = slug

        return super().save(*args, **kwargs)

    _root_layout_object = None

    @property
    def layout(self):
        if self._root_layout_object:
            return self._root_layout_object

        def transform(block, parent=None):
            block_object = get_block_class_by_type(block.get('type'))(
                uuid=block.get('uuid'),
                type=block.get('type'),
                attributes=block.get('attributes') or {},
                parent=parent,
            )
            block_object.children = [transform(x, parent=block_object) for x in block.get('children') or []]
            return block_object

        self._root_layout_object = transform(self._layout or {'type': 'root'})
        self._root_layout_object.cell_layout = self
        return self._root_layout_object
