# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.urls import reverse
from django.utils.html import format_html
from django.utils.translation import gettext_lazy as _

from combo.middleware import get_request

from ..utils.data_sources import get_data_source_from_dict


class ContentCellForm:
    def __init__(self, *args, **kwargs):
        from ..models import CellLayout

        super().__init__(*args, **kwargs)
        data_source = self.initial.get('_data_source')
        if get_request().user.is_staff:
            self.fields['layout'].help_text = format_html(
                '{} <a href="{}">{}</a>',
                _('In order to add and edit layouts:'),
                reverse('manager-layouts-index'),
                _('Layout management'),
            )
        self.fields['layout'].choices = [
            ('__list__', _('Simple list')),
        ]
        if data_source:
            self.fields['layout'].choices.extend(
                [
                    (x.uuid, x.title)
                    for x in CellLayout.objects.all().filter(
                        _data_source__provider_id=data_source.get('provider_id'),
                        _data_source__service_id=data_source.get('service_id'),
                        _data_source__slug=data_source.get('slug'),
                    )
                ]
            )
        field_order = ['_data_source']
        data_source_object = get_data_source_from_dict(data_source)
        if data_source_object:
            for k, v in data_source_object.get_selection_option_fields(cell=self.instance).items():
                self.initial[k] = getattr(data_source_object, k)
                self.fields[k] = v
                field_order.append(k)
        self.order_fields(field_order)

    def clean(self):
        super().clean()
        data_source_dict = self.cleaned_data.get('_data_source')
        data_source_object = get_data_source_from_dict(data_source_dict)
        for k, v in data_source_object.get_selection_option_fields(cell=self.instance).items():
            data_source_dict[k] = self.cleaned_data.get(k, v.initial)
