from unittest import mock

import pytest
from django.template import Context
from django.test.client import RequestFactory

from combo.apps.content.utils.layouts import LayoutBlockAttribute
from combo.apps.wcs.data_sources import (
    CardsDataSourceProvider,
    FormsDataSourceProvider,
    UserFormsDataSourceProvider,
)
from combo.utils import NothingInCacheException

from .utils import MockUserWithNameId, mocked_requests_send

pytestmark = pytest.mark.django_db


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_cards_data_source_provider(mock_send, app):
    provider = CardsDataSourceProvider()
    assert [x.slug for x in provider.choices if x.service_id == 'default'] == [
        'card_model_1',
        'card_model_2',
        'card_model_3',
        'card_a',
        'card_b',
        'card_c',
        'card_d',
        'card-e',
    ]
    data_source = [x for x in provider.choices if (x.service_id, x.slug) == ('default', 'card_model_1')][0]

    assert data_source.get_attribute_choices()[2] == ('Workflow Data', [('item', 'Item')])
    assert data_source.get_attribute_label('fielda') == 'Field A'
    assert data_source.get_attribute_label('metadata:status') == 'Status'
    assert data_source.get_attribute_label('') == '(unset)'

    ctx = {'request': RequestFactory().get('/')}

    with pytest.raises(NothingInCacheException):
        data_source.get_data(ctx)

    ctx['synchronous'] = True
    data_dict = data_source.get_data(ctx)
    assert data_dict.get('count') == 3
    assert [x.metadata.text for x in data_dict.get('data')] == ['aa', 'bb', 'cc']
    assert [x.fields.fielda for x in data_dict.get('data')] == ['<i>a</i>', '', '']

    data_source.custom_view = 'foo-view'
    data_dict = data_source.get_data(ctx)
    assert 'api/cards/card_model_1/list/foo-view?' in mock_send.call_args_list[-1][0][0].url
    data_source.custom_view = ''

    data_source.selection = 'url'
    data_dict = data_source.get_data(ctx)
    assert data_dict == {'data': [], 'count': 0, 'err': 1}

    ctx['card_model_1_id'] = '4'
    data_dict = data_source.get_data(ctx)
    assert data_dict == {'count': 0, 'data': [], 'err': 0}

    ctx['card_model_1_id'] = '11'
    data_dict = data_source.get_data(ctx)
    assert data_dict['count'] == 1
    assert data_dict['data'][0].metadata.id == '11'

    data_source.selection = 'template'
    data_source.ids_template = '12'
    data_dict = data_source.get_data(ctx)
    assert data_dict['count'] == 1
    assert data_dict['data'][0].metadata.id == '12'

    data_source.selection = 'template'
    data_source.ids_template = '{% %}'  # TemplateSyntaxError
    data_dict = data_source.get_data(ctx)
    assert data_dict['count'] == 0

    data_source.selection = 'all'
    data_dict = data_source.get_data(ctx, paginate_by=2, page_offset=0)
    assert len(data_dict['data']) == 2
    assert data_dict['count'] == 3

    data_dict = data_source.get_data(ctx, paginate_by=2, page_offset=2)
    assert len(data_dict['data']) == 1
    assert data_dict['count'] == 3

    data_source.linked_to_user = True
    ctx['request'].user = MockUserWithNameId()
    data_dict = data_source.get_data(ctx, paginate_by=2, page_offset=0)
    assert 'filter-user-uuid=xyz' in mock_send.call_args_list[-1][0][0].url

    data_source.selection = 'template'
    data_source.ids_template = '11'
    data_dict = data_source.get_data(ctx)
    assert data_dict['data'][0].metadata.status == 'Recorded'
    assert data_dict['data'][0].fields.item == 'foo'
    assert (
        str(data_dict['data'][0].fields.fielde)
        == '<p>lorem&lt;strong&gt;ipsum\n</p><p>\nhello&#x27;world</p>'
    )
    assert (
        str(data_dict['data'][0].fields.fieldf)
        == '<p class="plain-text-pre">lorem&lt;strong&gt;ipsum\n\nhello world</p>'
    )
    assert str(data_dict['data'][0].fields.fieldg) == 'test@localhost'
    assert str(data_dict['data'][0].fields.fieldi) == "<p>lorem<strong>ipsum</p><p>hello'world</p>"
    assert str(data_dict['data'][0].fields.fieldii) == "<p>lorem<strong>ipsum</p><p>hello'world</p>"
    assert str(data_dict['data'][0].fields.fieldd) == 'file.pdf'
    assert (
        data_dict['data'][0]
        .get_value('fieldd', context=Context({'request': RequestFactory().get('/')}))
        .startswith('<a class="pk-card-field-filename"')
    )
    assert (
        data_dict['data'][0].get_value('field_image', context=Context({'request': RequestFactory().get('/')}))
        == '<img src="http://127.0.0.1:8999/download?f=45&thumbnail=1" alt="">'
    )

    assert data_dict['data'][0].get_value('fieldg') == 'test@localhost'
    assert (
        data_dict['data'][0].get_value(
            'fieldg', layout_block=LayoutBlockAttribute(attributes={'email_display_mode': 'link'})
        )
        == '<a href="mailto:test@localhost">test@localhost</a>'
    )

    assert data_dict['data'][0].title() == 'aa'
    assert data_dict['data'][0].url() == 'http://127.0.0.1:8999/backoffice/data/card_model_1/11/'


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_forms_data_source_provider(mock_send, app):
    provider = FormsDataSourceProvider()
    assert [x.slug for x in provider.choices if x.service_id == 'default'] == [
        'a-private-form',
        'a-second-form-title',
        'form-title',
        'third-form-title',
    ]
    data_source = [x for x in provider.choices if (x.service_id, x.slug) == ('default', 'a-private-form')][0]

    ctx = {'request': RequestFactory().get('/')}
    ctx['synchronous'] = True
    data_dict = data_source.get_data(ctx)
    assert data_dict['count'] == 2

    data_source.custom_view = 'foo-view'
    data_dict = data_source.get_data(ctx)
    assert '/api/forms/a-private-form/list/foo-view?' in mock_send.call_args_list[-1][0][0].url


@mock.patch('requests.Session.send', side_effect=mocked_requests_send)
def test_user_forms_data_source_provider(mock_send, app):
    provider = UserFormsDataSourceProvider()
    assert len(provider.choices) == 2  # one per wcs-site
    data_source = provider.choices[0]
    ctx = {'request': RequestFactory().get('/'), 'synchronous': True}
    data_dict = data_source.get_data(ctx)
    assert data_dict['count'] == 3
    assert [x.url() for x in data_source.get_data(ctx)['data']] == [
        'http://127.0.0.1:8999/form/1/',
        'http://127.0.0.1:8999/form/2/',
        'http://127.0.0.1:8999/form/3/',
    ]

    # only drafts, will use /api/user/drafts
    data_source.include_open_forms = False
    data_source.include_closed_forms = False
    data_source.include_drafts = True
    data_dict = data_source.get_data(ctx)
    assert data_dict['count'] == 4

    # fake pagination
    data_dict = data_source.get_data(ctx, paginate_by=3, page_offset=1)
    assert len(data_dict['data']) == 3
    assert data_dict['count'] == 4

    data_dict = data_source.get_data(ctx, paginate_by=3, page_offset=3)
    assert len(data_dict['data']) == 1
    assert data_dict['count'] == 4

    # concerned user
    ctx['request'].user = MockUserWithNameId()
    data_dict = data_source.get_data(ctx)
    assert '/api/users/xyz/drafts' in mock_send.call_args_list[-1][0][0].url

    data_source.include_open_forms = True
    data_source.include_closed_forms = True
    data_source.include_drafts = False
    data_dict = data_source.get_data(ctx)
    assert 'status=all' in mock_send.call_args_list[-1][0][0].url

    data_source.include_open_forms = False
    data_dict = data_source.get_data(ctx)
    assert 'status=done' in mock_send.call_args_list[-1][0][0].url

    data_source.include_open_forms = True
    data_source.include_closed_forms = False
    data_dict = data_source.get_data(ctx)
    assert 'status=open' in mock_send.call_args_list[-1][0][0].url

    data_source.include_forms_user_can_access = True
    data_dict = data_source.get_data(ctx)
    assert 'include-accessible=on' in mock_send.call_args_list[-1][0][0].url
