# combo - content management system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import dataclasses
import datetime
import urllib.parse

import requests
from django import forms
from django.conf import settings
from django.utils.timezone import localtime
from django.utils.translation import gettext_lazy as _

from combo import utils
from combo.data.library import register_cell_class
from combo.data.models import UniverCell


def get_publik_service(service):
    services = settings.KNOWN_SERVICES.get(service, {})

    for service in services.values():
        if not service.get('secondary', False):
            return service


@dataclasses.dataclass
class Event:
    slug: str
    start_datetime: datetime.datetime
    end_datetime: datetime.datetime
    description: str
    pricing: str
    url: str
    label: str
    color: str
    full: bool
    disabled: bool

    @property
    def date(self):
        return self.start_datetime.date()

    @property
    def height(self):
        return (self.end_datetime - self.start_datetime).seconds / 3600

    @staticmethod
    def parse_datetime(datetime_str):
        return datetime.datetime.strptime(datetime_str, '%Y-%m-%d %H:%M:%S')

    @classmethod
    def get_events_from_json(cls, data):
        return [
            cls(
                slug=event['slug'],
                start_datetime=cls.parse_datetime(event['datetime']),
                end_datetime=cls.parse_datetime(event['end_datetime']),
                description=event['description'],
                pricing=event['pricing'],
                url=event['url'],
                label=event['label'],
                color=event.get('custom_field_color'),
                full=event['places']['full'],
                disabled=event['disabled'],
            )
            for event in data
            if event['end_datetime']
        ]


@register_cell_class
class EventsCalendarCell(UniverCell):
    default_template_name = 'combo/chrono/events-calendar.html'

    class Meta:
        proxy = True
        verbose_name = _('Weekly events calendar')

    class Media:
        js = ('js/combo.events-calendar.js', 'xstatic/jquery-ui.min.js')
        css = {'all': ('css/combo.events-calendar.css',)}

    @classmethod
    def get_attribute_fields(cls):
        return {
            'title': forms.CharField(label=_('Title'), max_length=200, required=False),
            'agendas': forms.CharField(
                label=_('Agenda slugs'),
                max_length=1000,
                required=False,
                initial='',
                help_text=_('Comma separated list of agenda slugs.'),
                widget=forms.TextInput(attrs={'class': 'text-wide'}),
            ),
            'booking_form_slug': forms.CharField(
                label=_('Booking form identifier'),
                max_length=250,
                required=False,
                initial='',
                widget=forms.TextInput(attrs={'class': 'text-wide'}),
            ),
            'hour_height': forms.IntegerField(
                label=_('Hour height (px)'),
                initial=80,
                min_value=10,
                max_value=500,
            ),
        }

    def get_page_absolute_url(self):
        return urllib.parse.urljoin(settings.SITE_BASE_URL, self.page.get_online_url())

    @property
    def booking_form_url(self):
        wcs = get_publik_service('wcs')
        return urllib.parse.urljoin(wcs['url'], self.booking_form_slug) + '/'

    @property
    def chrono_url(self):
        chrono = get_publik_service('chrono')
        return urllib.parse.urljoin(chrono['url'], 'api/agendas/datetimes/')

    def get_url_params(self):
        today = localtime().date()
        monday = today - datetime.timedelta(days=today.weekday())

        return {
            'agendas': self.agendas,
            'date_start': monday.isoformat(),
            'show_past_events': True,
        }

    def get_events_json(self):
        try:
            response = utils.requests.get(
                self.chrono_url,
                params=self.get_url_params(),
                headers={'Accept': 'application/json'},
                remote_service='auto',
                without_user=True,
            )
            response.raise_for_status()
        except requests.RequestException:
            return []

        try:
            events_json = response.json()
        except ValueError:
            return []

        return events_json['data']

    def add_events_context(self, events, request, extra_context):
        week = request.GET.get('week')
        if not week:
            first_date = events[0].date if events else localtime().date()
            current_monday = first_date - datetime.timedelta(days=first_date.weekday())
        else:
            current_monday = datetime.date.fromisoformat(week)

        previous_monday = current_monday - datetime.timedelta(days=7)
        next_monday = current_monday + datetime.timedelta(days=7)

        if events:
            min_time = min(x.start_datetime.time() for x in events)
            max_time = max(x.end_datetime.time() for x in events)
            extra_context['hours'] = [
                datetime.time(hour, 0) for hour in range(min_time.hour, max_time.hour + 1)
            ]

        events_by_day = {current_monday + datetime.timedelta(days=i): [] for i in range(5)}
        has_past_events = has_future_events = False
        for event in events:
            if event.date < current_monday:
                has_past_events = True
                continue
            if event.date >= next_monday:
                has_future_events = True
                break

            if event.date not in events_by_day:
                if event.start_datetime.weekday() == 5:
                    # add Saturday
                    events_by_day[event.date] = []
                elif event.start_datetime.weekday() == 6:
                    # add Saturday and Sunday
                    events_by_day.setdefault(event.date - datetime.timedelta(days=1), [])
                    events_by_day[event.date] = []

            min_datetime = datetime.datetime.combine(event.start_datetime, min_time)
            event.top = (event.start_datetime - min_datetime).seconds / 3600

            events_by_day[event.date].append(event)

        extra_context.update(
            {
                'days': list(events_by_day),
                'events_by_day': events_by_day,
                'previous_monday': previous_monday if has_past_events else None,
                'next_monday': next_monday if has_future_events else None,
            }
        )

    def get_cell_extra_context(self, context, invalidate_cache=False):
        extra_context = super().get_cell_extra_context(context)

        events_json = self.get_events_json()
        events = Event.get_events_from_json(events_json)
        self.add_events_context(events, context['request'], extra_context)

        return extra_context
