# combo - content management system
# Copyright (C) 2014-2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.db import connection, migrations
from django.utils.module_loading import import_string


class UnivercellMigration(migrations.RunPython):
    bulk_create = True

    def __init__(self, app_name, dotted_class_name):
        super().__init__(self.forward, self.backward)
        self.app_name = app_name
        self.dotted_class_name = dotted_class_name

    def forward(self, apps, schema_editor):
        klass_name = self.dotted_class_name.split('.')[-1]
        klass = apps.get_model(self.app_name, f'Old{klass_name}')
        new_class = apps.get_model(self.app_name, klass_name)
        real_class = import_string(self.dotted_class_name)
        content_type_class = apps.get_model('contenttypes', 'ContentType')
        self.old_content_type_id = content_type_class.objects.get_for_model(klass).id
        self.new_content_type_id = content_type_class.objects.get_for_model(new_class).id
        try:
            self.tile_class = apps.get_model('dashboard', 'Tile')
        except LookupError:
            self.tile_class = None

        new_objects = []
        for instance in klass.objects.all().prefetch_related('groups'):
            new_objects.append(
                {'old_id': instance.id, 'instance': self.forward_instance(new_class, real_class, instance)}
            )

        if self.bulk_create:
            new_class.objects.bulk_create(
                [x['instance'] for x in new_objects if not hasattr(x['instance'], '_do_not_bulk_save')]
            )

        if self.tile_class:
            for new_object in new_objects:
                self.tile_class.objects.filter(
                    cell_type_id=self.old_content_type_id, cell_pk=new_object.get('old_id')
                ).update(
                    cell_type_id=self.new_content_type_id,
                    cell_pk=new_object['instance'].id,
                )

    def forward_instance_attributes(self, real_class, instance):
        instance.attributes = {k: getattr(instance, k) for k in real_class.get_attribute_fields()}

    def forward_instance(self, new_class, real_class, instance):
        instance_groups = instance.groups.all()
        instance.id = None
        instance.__class__ = new_class
        self.forward_instance_attributes(real_class, instance)
        instance.class_name = self.dotted_class_name
        if instance_groups or not self.bulk_create:
            instance.save()
            if instance_groups:
                instance = new_class.objects.get(id=instance.id)
                instance._do_not_bulk_save = True
                instance.groups.set(instance_groups)
        return instance

    def backward(self, apps, schema_editor):
        # bulk remove
        # (tile classes are not supported in backward migrations)
        with connection.cursor() as cursor:
            cursor.execute(
                '''DELETE FROM data_univercell_groups
                               WHERE univercell_id IN (
                                 SELECT id FROM data_univercell WHERE class_name = %s)''',
                (self.dotted_class_name,),
            )
            cursor.execute('DELETE FROM data_univercell WHERE class_name = %s', (self.dotted_class_name,))
