#
# combo - content management system
# Copyright (C) 2015-2017  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from functools import partial

import ckeditor.fields
from bleach import Cleaner, linkifier
from bleach.css_sanitizer import CSSSanitizer
from django import forms
from django.conf import settings
from django.core import validators
from django.forms.widgets import TextInput
from django.utils.encoding import force_str


class RichTextField(ckeditor.fields.RichTextField):
    def formfield(self, **kwargs):
        defaults = {
            'form_class': RichTextFormField,
            'config_name': self.config_name,
            'extra_plugins': self.extra_plugins,
            'external_plugin_resources': self.external_plugin_resources,
        }
        defaults.update(kwargs)
        return super().formfield(**defaults)


class RichTextFormField(ckeditor.fields.RichTextFormField):
    ALL_TAGS = [
        'a',
        'abbr',
        'acronym',
        'address',
        'area',
        'article',
        'aside',
        'audio',
        'b',
        'big',
        'blockquote',
        'br',
        'button',
        'canvas',
        'caption',
        'center',
        'cite',
        'code',
        'col',
        'colgroup',
        'command',
        'datagrid',
        'datalist',
        'dd',
        'del',
        'details',
        'dfn',
        'dialog',
        'dir',
        'div',
        'dl',
        'dt',
        'em',
        'event-source',
        'fieldset',
        'figcaption',
        'figure',
        'font',
        'footer',
        'form',
        'h1',
        'h2',
        'h3',
        'h4',
        'h5',
        'h6',
        'header',
        'hr',
        'i',
        'img',
        'input',
        'ins',
        'kbd',
        'keygen',
        'label',
        'legend',
        'li',
        'm',
        'map',
        'menu',
        'meter',
        'multicol',
        'nav',
        'nextid',
        'noscript',
        'ol',
        'optgroup',
        'option',
        'output',
        'p',
        'pre',
        'progress',
        'q',
        's',
        'samp',
        'section',
        'select',
        'small',
        'sound',
        'source',
        'spacer',
        'span',
        'strike',
        'strong',
        'sub',
        'summary',
        'sup',
        'table',
        'tbody',
        'td',
        'textarea',
        'tfoot',
        'th',
        'thead',
        'time',
        'tr',
        'tt',
        'u',
        'ul',
        'var',
        'video',
    ]
    ALL_ATTRS = [
        'abbr',
        'accept',
        'accept-charset',
        'accesskey',
        'action',
        'align',
        'alt',
        'aria-label',
        'aria-level',
        'autocomplete',
        'autofocus',
        'axis',
        'background',
        'balance',
        'bgcolor',
        'bgproperties',
        'border',
        'bordercolor',
        'bordercolordark',
        'bordercolorlight',
        'bottompadding',
        'cellpadding',
        'cellspacing',
        'ch',
        'challenge',
        'char',
        'charoff',
        'charset',
        'checked',
        'choff',
        'cite',
        'class',
        'clear',
        'color',
        'cols',
        'colspan',
        'compact',
        'contenteditable',
        'controls',
        'coords',
        'data',
        'datafld',
        'datapagesize',
        'datasrc',
        'datetime',
        'default',
        'delay',
        'dir',
        'disabled',
        'draggable',
        'dynsrc',
        'enctype',
        'end',
        'face',
        'for',
        'form',
        'frame',
        'galleryimg',
        'gutter',
        'headers',
        'height',
        'hidden',
        'hidefocus',
        'high',
        'href',
        'hreflang',
        'hspace',
        'icon',
        'id',
        'inputmode',
        'ismap',
        'keytype',
        'label',
        'lang',
        'leftspacing',
        'list',
        'longdesc',
        'loop',
        'loopcount',
        'loopend',
        'loopstart',
        'low',
        'lowsrc',
        'max',
        'maxlength',
        'media',
        'method',
        'min',
        'multiple',
        'name',
        'nohref',
        'noshade',
        'nowrap',
        'open',
        'optimum',
        'pattern',
        'ping',
        'point-size',
        'poster',
        'pqg',
        'preload',
        'prompt',
        'radiogroup',
        'readonly',
        'rel',
        'repeat-max',
        'repeat-min',
        'replace',
        'required',
        'rev',
        'rightspacing',
        'role',
        'rows',
        'rowspan',
        'rules',
        'scope',
        'selected',
        'shape',
        'size',
        'span',
        'src',
        'start',
        'step',
        'style',
        'summary',
        'suppress',
        'tabindex',
        'target',
        'template',
        'title',
        'toppadding',
        'type',
        'unselectable',
        'urn',
        'usemap',
        'valign',
        'value',
        'variable',
        'volume',
        'vrml',
        'vspace',
        'width',
        'wrap',
        'xml:lang',
    ]
    ALL_STYLES = [
        'azimuth',
        'background-color',
        'border-bottom-color',
        'border-bottom-left-radius',
        'border-bottom-right-radius',
        'border-collapse',
        'border-color',
        'border-left-color',
        'border-radius',
        'border-right-color',
        'border-top-color',
        'border-top-left-radius',
        'border-top-right-radius',
        'clear',
        'color',
        'cursor',
        'direction',
        'display',
        'elevation',
        'float',
        'font',
        'font-family',
        'font-size',
        'font-style',
        'font-variant',
        'font-weight',
        'height',
        'letter-spacing',
        'line-height',
        'margin',
        'margin-bottom',
        'margin-left',
        'margin-right',
        'margin-top',
        'overflow',
        'padding',
        'padding-bottom',
        'padding-left',
        'padding-right',
        'padding-top',
        'pause',
        'pause-after',
        'pause-before',
        'pitch',
        'pitch-range',
        'richness',
        'speak',
        'speak-header',
        'speak-numeral',
        'speak-punctuation',
        'speech-rate',
        'stress',
        'text-align',
        'text-decoration',
        'text-indent',
        'unicode-bidi',
        'vertical-align',
        'voice-family',
        'volume',
        'white-space',
        'width',
    ]
    TLDS = [x for x in linkifier.TLDS if x != 'id']
    URL_RE = linkifier.build_url_re(tlds=TLDS)
    EMAIL_RE = linkifier.build_email_re(tlds=TLDS)

    def clean(self, value):
        value = super().clean(value)

        if settings.CKEDITOR_CLEAN_HTML:
            allowed_tags = self.ALL_TAGS[:]
            if settings.CKEDITOR_ALLOW_STYLE_TAGS:
                allowed_tags.append('style')
            if settings.CKEDITOR_ALLOW_SCRIPT_TAGS:
                allowed_tags.append('script')
            if settings.CKEDITOR_ALLOW_LINK_TAGS:
                allowed_tags.append('link')
            cleaner = Cleaner(
                tags=allowed_tags,
                css_sanitizer=CSSSanitizer(allowed_css_properties=self.ALL_STYLES),
                attributes=self.ALL_ATTRS,
                strip=True,
                strip_comments=False,
                filters=[
                    partial(
                        linkifier.LinkifyFilter,
                        skip_tags=['pre'],
                        parse_email=True,
                        url_re=self.URL_RE,
                        email_re=self.EMAIL_RE,
                    )
                ],
            )
            value = cleaner.clean(value).removeprefix('<br />').removesuffix('<br />')

        if settings.LANGUAGE_CODE.startswith('fr-'):
            # apply some typographic rules
            value = value.replace('&laquo; ', '«\u202f')
            value = value.replace('« ', '«\u202f')
            value = value.replace(' &raquo;', '\u202f»')
            value = value.replace(' »', '\u202f»')
            value = value.replace(' :', '\u00a0:')
            value = value.replace(' ;', '\u202f;')
            value = value.replace(' !', '\u202f!')
            value = value.replace(' ?', '\u202f?')
        return value


def templatable_url_validator(value):
    value = force_str(value)
    if '{{' in value or '{%' in value:
        # leave templates alone
        return
    validators.URLValidator()(value)


class TemplatableURLField(forms.URLField):
    widget = TextInput
    default_validators = [templatable_url_validator]

    def to_python(self, value):
        value = super(forms.URLField, self).to_python(value)
        if '{{' in value or '{%' in value:
            return value
        return super().to_python(value)
