/* global gettext */

window.addEventListener('DOMContentLoaded', () => {
  const optionsPrefix = document.querySelector('.appearance-screen--options').dataset.optionsPrefix
  const saveStatus = document.getElementById('save-status')
  let saveStatusTimeout = null

  function computeForegroundColors () {
    document.querySelectorAll('.auto-color').forEach(
      function (input) {
        let color = document.querySelector('input.auto-color').previousElementSibling.value
        if (color) {
          let brightness = parseInt(color.slice(1, 3), 16) * 0.212671
              + parseInt(color.slice(3, 5), 16) * 0.715160
              + parseInt(color.slice(5, 7), 16) * 0.072169
          if (brightness > 128) {
            input.value = '#000000'
          } else {
            input.value = '#FFFFFF'
          }
        }
      },
    )
  }

  function getRelativeLuminance (color) {
    let RsRGB = parseInt(color.slice(1, 3), 16) / 255
    let GsRGB = parseInt(color.slice(3, 5), 16) / 255
    let BsRGB = parseInt(color.slice(5, 7), 16) / 255
    let R = RsRGB < 0.04045 ? RsRGB / 12.92 : ((RsRGB + 0.055) / 1.055) ** 2.4
    let G = GsRGB < 0.04045 ? GsRGB / 12.92 : ((GsRGB + 0.055) / 1.055) ** 2.4
    let B = GsRGB < 0.04045 ? BsRGB / 12.92 : ((BsRGB + 0.055) / 1.055) ** 2.4
    let L = 0.2126 * R + 0.7152 * G + 0.0722 * B
    return L
  }

  function computeContrastInfo () {
    document.querySelectorAll('.contrast-info').forEach(
      function (contrastInfo) {
        let fgColor = contrastInfo.parentNode.querySelector('[type="color"]').value
        let bgColor = contrastInfo.querySelector('.contrast-info--background').dataset.bg
        if (!bgColor.startsWith('#')) {
          bgColor = document.querySelector(`[name="${bgColor}"]`).value
        }
        contrastInfo.querySelector('.contrast-info--foreground').style.backgroundColor = fgColor
        contrastInfo.querySelector('.contrast-info--background').style.backgroundColor = bgColor
        let fgRelativeLuminance = getRelativeLuminance(fgColor)
        let bgRelativeLuminance = getRelativeLuminance(bgColor)
        let lighterLuminance = (fgRelativeLuminance > bgRelativeLuminance
          ? fgRelativeLuminance : bgRelativeLuminance)
        let darkerLuminance = (fgRelativeLuminance < bgRelativeLuminance
          ? fgRelativeLuminance : bgRelativeLuminance)
        let contrastRatio = (lighterLuminance + 0.05) / (darkerLuminance + 0.05)
        contrastInfo.querySelector('.contrast-info--value').textContent
                      = contrastRatio.toLocaleString(undefined, {maximumFractionDigits: 2})
        let mark = contrastInfo.querySelector('.contrast-info--mark')
        if (contrastRatio < 4.5) {
          mark.textContent = '⚠️'
          mark.classList.remove('contrast-info--ok')
          mark.classList.add('contrast-info--ko')
          contrastInfo.setAttribute('title', gettext('Inadequate contrast for text accessibility'))
        } else if (contrastRatio < 7) {
          mark.textContent = 'AA✓'
          mark.classList.remove('contrast-info--ko')
          mark.classList.add('contrast-info--ok')
          contrastInfo.setAttribute('title', gettext('Adequate contrast for text accessibility (AA level)'))
        } else {
          mark.textContent = 'AAA✓'
          mark.classList.remove('contrast-info--ko')
          mark.classList.add('contrast-info--ok')
          contrastInfo.setAttribute('title', gettext('Adequate contrast for text accessibility (AAA level)'))
        }
      },
    )
  }

  document.querySelector('.appearance-screen--options--form').addEventListener('submit',
    async function (event) {
      event.preventDefault()
      const data = new FormData(this)
      try {
        const resp = await fetch(this.attributes.action.value, {method: 'POST', body: data})
        const result = await resp.json()
        if (result.err === 0) {
          saveStatus.innerText = gettext('Options saved.')
        } else {
          saveStatus.innerText = gettext('Error saving options.')
        }
      } catch (err) {
        saveStatus.innerText = gettext('Unknown error saving options.')
      }
      saveStatus.style.display = 'block'
      clearTimeout(saveStatusTimeout)
      saveStatusTimeout = setTimeout(() => { saveStatus.style.display = 'none' }, 5000)
    },
  )

  document.getElementById('switch-mobile-view').addEventListener('click', function () {
    document.querySelector('.appearance-screen--preview-content').classList.remove(
      'appearance-screen--preview-content-desktop')
    document.querySelector('.appearance-screen--preview-content').classList.add(
      'appearance-screen--preview-content-mobile')
    document.querySelector('.appearance-screen--preview-width .active').classList.remove('active')
    this.classList.add('active')
  })
  document.getElementById('switch-desktop-view').addEventListener('click', function () {
    document.querySelector('.appearance-screen--preview-content').classList.remove(
      'appearance-screen--preview-content-mobile')
    document.querySelector('.appearance-screen--preview-content').classList.add(
      'appearance-screen--preview-content-desktop')
    document.querySelector('.appearance-screen--preview-width .active').classList.remove('active')
    this.classList.add('active')
  })

  let styles = null
  const iframe = document.querySelector('iframe')

  iframe.addEventListener('load', function () {
    styles = iframe.contentDocument.createElement('style')
    iframe.contentDocument.head.appendChild(styles)
    updateStyles()
  })

  function updateStyles () {
    computeForegroundColors()
    computeContrastInfo()
    let options = []
    document.querySelectorAll('.appearance-screen--options table [name]').forEach(
      function (input) {
        if (input.name === 'csrfmiddlewaretoken') return
        if (input.value.indexOf(' ') > -1) {
          options.push(`--${optionsPrefix}-${input.name}: "${input.value}";`)
        } else {
          options.push(`--${optionsPrefix}-${input.name}: ${input.value};`)
        }
      },
    )
    styles.innerHTML = `html { ${options.join(' ')} }`
  }

  // init
  document.querySelectorAll('.appearance-screen--options table [name]').forEach(
    function (input) {
      if (input.name === 'csrfmiddlewaretoken') return
      input.dataset.currentValue = input.value
    },
  )

  function valueChange (ev) {
    const input = ev.target
    let previousValue = input.dataset.currentValue
    updateStyles()
    if (previousValue && input.value !== previousValue) {
      // update to new value
      input.dataset.currentValue = input.value

      let elem = input
      while (elem && elem.nodeName !== 'TR') { elem = elem.parentNode } // get <tr> ancestor
      const optionTr = elem
      const optionName = optionTr.dataset.optionName

      let previousHeight; let previousWidth; let newHeight; let newWidth

      if (optionTr.querySelector('.input-size')) {
        const thisInputWidth = optionTr.querySelector('.input-size--width')
        const thisInputHeight = optionTr.querySelector('.input-size--height')
        if (input === thisInputHeight) {
          previousHeight = previousValue
          previousWidth = thisInputWidth.value
          newHeight = input.value
          newWidth = previousWidth
        } else {
          previousHeight = thisInputHeight.value
          previousWidth = previousValue
          newHeight = previousHeight
          newWidth = input.value
        }
      }

      // update inherited variables
      document.querySelectorAll(`[data-default-same-as="${optionName}"]`).forEach(
        function (otherOptionLine) {
          if (otherOptionLine.querySelector('.input-size')) {
            const otherInputHeight = otherOptionLine.querySelector('.input-size--height')
            const otherInputWidth = otherOptionLine.querySelector('.input-size--width')
            if (otherInputHeight.value === previousHeight && otherInputWidth.value === previousWidth) {
              otherInputHeight.value = newHeight
              otherInputWidth.value = newWidth
              otherInputWidth.dispatchEvent(new Event('change'))
              updateStyles()
            }
          } else {
            let otherInput = otherOptionLine.querySelector('input, select')
            if (otherInput.value === previousValue) {
              otherInput.value = input.value
              otherInput.dispatchEvent(new Event('change'))
              updateStyles()
            }
          }
        },
      )
    }
  }

  document.querySelectorAll('.appearance-screen--options table [name]').forEach(
    function (input) {
      ['change', 'input', 'paste'].forEach((ev) => input.addEventListener(ev, valueChange))
    },
  )

  document.querySelector('.appearance-screen--options--filter input').addEventListener('input', function () {
    let val = this.value.toLowerCase()
    document.querySelectorAll('table tbody tr').forEach(function (tr) {
      if (tr.querySelector('th')) {
        if (val) {
          tr.style.display = 'none'
        } else {
          tr.style.display = 'table-row'
        }
      } else if (tr.querySelector('td').textContent.toLowerCase().indexOf(val) === -1) {
        tr.style.display = 'none'
      } else {
        tr.style.display = 'table-row'
      }
    })
  })
})
