import pytest
import responses
from django.conf import settings
from django.test import override_settings

from combo.apps.appearance.manager_views import ThemeOption
from combo.data.models import Page

from .test_manager import login

pytestmark = pytest.mark.django_db


@override_settings(APPEARANCE_SUPPORT_ENABLED=True)
def test_appearance_manager(app, admin_user):
    Page.objects.create(title='Page', slug='index')
    app = login(app)
    with override_settings(APPEARANCE_SUPPORT_ENABLED=False):
        resp = app.get('/manage/', status=200)
        assert 'Appearance' not in resp.text

    resp = app.get('/manage/', status=200)
    assert 'Appearance' in resp.text
    resp = resp.click('Appearance')
    assert 'There are no options.' in resp.text

    with override_settings(
        APPEARANCE_OPTIONS={
            'button_color': {'default_same_as': 'primary_color', 'label': 'Button color', 'type': 'color'},
            'cell_entry_color': {
                'default_same_as': 'link_color',
                'label': 'Cell entry color',
                'type': 'color',
            },
            'link_color': {'default_same_as': 'primary_color', 'label': 'Link color', 'type': 'color'},
            'primary_color': {'default': '#117187', 'label': 'Primary color', 'rank': 0, 'type': 'color'},
            'logo_size': {
                'default': {'height': '100px', 'width': '200px'},
                'label': 'Logo size',
                'type': 'size',
            },
            'title_weight': {
                'default': '400',
                'help_text': '(page title)',
                'label': 'Title weight',
                'options': [
                    {'label': 'Regular', 'value': '400'},
                    {'label': 'Bold', 'value': '600'},
                ],
                'type': 'select',
            },
        }
    ):
        resp = app.get('/manage/appearance/')
        # check all options are there
        assert {x.attrib.get('data-option-name') for x in resp.pyquery('tbody tr')} == set(
            settings.APPEARANCE_OPTIONS.keys()
        )
        # check there are no sections
        assert not resp.pyquery('tbody th')
        # check first option is "primary color"
        assert resp.pyquery('tbody tr')[0].attrib.get('data-option-name') == 'primary_color'
        # check option value
        assert resp.pyquery('input[name="primary_color"]').val() == '#117187'
        # check inherit options values
        assert resp.pyquery('input[name="link_color"]').val() == '#117187'
        assert resp.pyquery('input[name="cell_entry_color"]').val() == '#117187'
        # check size type
        assert resp.pyquery('input[name="logo_height"]').val() == '100px'
        assert resp.pyquery('input[name="logo_width"]').val() == '200px'
        # check select type
        assert [
            (x.attrib.get('value'), x.text) for x in resp.pyquery('select[name="title_weight"] option')
        ] == [('400', 'Regular'), ('600', 'Bold')]

        # change a value and submit
        resp.forms[1]['link_color'] = '#ff0000'
        resp_json = resp.forms[1].submit()
        assert resp_json.json == {'err': 0}
        resp = app.get('/manage/appearance/')
        assert resp.pyquery('input[name="link_color"]').val() == '#ff0000'


@override_settings(APPEARANCE_SUPPORT_ENABLED=True)
def test_appearance_manager_contrast(app, admin_user):
    app = login(app)
    with override_settings(
        APPEARANCE_OPTIONS={
            'primary_color': {
                'default': '#117187',
                'label': 'Primary color',
                'rank': 0,
                'type': 'color',
                'check_contrast': '#000000',
            },
            'other_color': {
                'default': '#00FF00',
                'label': 'Other color',
                'type': 'color',
                'check_contrast': 'primary_color',
            },
            'bg_color': {
                'default': '#00FF00',
                'label': 'Background color',
                'type': 'color',
                'check_contrast_inv': '#000000',
            },
        }
    ):
        resp = app.get('/manage/appearance/')
        assert resp.pyquery('[data-option-name="primary_color"] .contrast-info [data-bg="#000000"]')
        assert resp.pyquery('[data-option-name="other_color"] .contrast-info [data-bg="primary_color"]')
        assert resp.pyquery(
            '[data-option-name="bg_color"] .contrast-info.contrast-info--inv [data-bg="#000000"]'
        )


def test_theme_option_repr():
    option = ThemeOption(name='primary_color', label='Primary color', default='#0000ff')
    assert repr(option) == '<ThemeOption primary_color>'


@responses.activate
@override_settings(
    APPEARANCE_SUPPORT_ENABLED=True,
    APPEARANCE_OPTIONS={
        'primary_color': {
            'default': '#117187',
            'label': 'Primary color',
            'type': 'color',
            'save_in_hobo_as': 'theme_color',
        },
    },
    KNOWN_SERVICES={
        'hobo': {
            'hobo': {
                'title': 'test',
                'url': 'http://hobo/',
                'secret': 'hobo',
                'orig': 'hobo',
                'secondary': None,
            }
        }
    },
)
def test_appearance_save_in_hobo(app, admin_user, caplog):
    app = login(app)
    resp = app.get('/manage/appearance/', status=200)

    responses.post('http://hobo/api/variables/', json={'err': 0})

    resp.forms[1]['primary_color'] = '#ff0000'
    resp_json = resp.forms[1].submit()
    assert resp_json.json == {'err': 0}

    # check with error
    responses.post('http://hobo/api/variables/', json={'err': 1}, status=400)

    resp.forms[1]['primary_color'] = '#ff0000'
    resp_json = resp.forms[1].submit()
    assert resp_json.json == {'err': 0}
    assert 'error sending appearance variable to hobo' in [x.message for x in caplog.records]


@override_settings(APPEARANCE_SUPPORT_ENABLED=True)
def test_appearance_manager_sections(app, admin_user):
    app = login(app)
    with override_settings(
        APPEARANCE_OPTIONS={
            'button_color': {
                'section': 'Colors',
                'default_same_as': 'primary_color',
                'label': 'Button color',
                'type': 'color',
            },
            'cell_entry_color': {
                'section': 'Colors',
                'default_same_as': 'link_color',
                'label': 'Cell entry color',
                'type': 'color',
            },
            'link_color': {
                'section': 'Colors',
                'default_same_as': 'primary_color',
                'label': 'Link color',
                'type': 'color',
            },
            'primary_color': {
                'section': 'General options',
                'default': '#117187',
                'label': 'Primary color',
                'rank': 0,
                'type': 'color',
            },
            'logo_size': {
                'default': {'height': '100px', 'width': '200px'},
                'label': 'Logo size',
                'type': 'size',
            },
            'title_weight': {
                'default': '400',
                'help_text': '(page title)',
                'label': 'Title weight',
                'options': [
                    {'label': 'Regular', 'value': '400'},
                    {'label': 'Bold', 'value': '600'},
                ],
                'type': 'select',
            },
        }
    ):
        resp = app.get('/manage/appearance/')
        assert [x.text for x in resp.pyquery('tbody th')] == ['General options', 'Colors', 'Other options']
