# combo - content management system
# Copyright (C) 2015-2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import json
import logging

from django.conf import settings
from django.http import JsonResponse
from django.utils.translation import gettext_lazy as _
from django.views.generic import TemplateView
from requests.exceptions import RequestException

from combo.utils import requests

from .models import AppearanceSettings


class ThemeOption:
    _value = None
    name = None
    rank = 1000
    save_in_hobo_as = None
    section = None

    def __init__(self, **kwargs):
        for k, v in kwargs.items():
            setattr(self, k, v)

        options_values = AppearanceSettings.get_options()
        if options_values:
            if self.type == 'size':
                height_name = self.name.removesuffix('size') + 'height'
                width_name = self.name.removesuffix('size') + 'width'
                if height_name in options_values and width_name in options_values:
                    self.value = {
                        'height': options_values.get(height_name),
                        'width': options_values.get(width_name),
                    }
            elif self.name in options_values:
                self.value = options_values[self.name]

        if not self.section:
            self.section = _('Other options')

    def __repr__(self):
        return f'<{self.__class__.__name__} {self.name}>'

    @property
    def value(self):
        return self._value or self.default

    @value.setter
    def value(self, value):
        self._value = value


def get_options():
    available_options = settings.APPEARANCE_OPTIONS or {}

    options = {x: ThemeOption(name=x, **y) for x, y in available_options.items()}
    sections = {x.section: 1000 for x in options.values()}
    for option in options.values():
        option_with_default = option
        while getattr(option_with_default, 'default_same_as', None):
            option_with_default = options[option_with_default.default_same_as]
        option.default = option_with_default.default
        sections[option.section] = min(sections[option.section], option.rank)

    sections[_('Other options')] = 1000
    ordered_sections = sorted(sections.keys(), key=sections.get)
    for option in options.values():
        option.rank = ordered_sections.index(option.section) * 10000 + option.rank

    if len(sections) == 1:
        # do not display section if there's a single one
        for option in options.values():
            option.section = None

    return options


class ManagerHomeView(TemplateView):
    template_name = 'combo/appearance/manager_home.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['options'] = list(get_options().values())
        context['options'].sort(key=lambda x: (x.rank, x.label))
        context['options_prefix'] = settings.APPEARANCE_OPTIONS_PREFIX
        context['iframe_url'] = settings.APPEARANCE_PREVIEW_IFRAME_URL
        return context


def save_options_view(request, *args, **kwargs):
    options = {x: y for x, y in request.POST.items() if x != 'csrfmiddlewaretoken'}

    appearance_settings = AppearanceSettings.singleton()
    appearance_settings.options = options
    appearance_settings.save()

    try:
        hobo_service = [x for x in settings.KNOWN_SERVICES['hobo'].values() if x.get('secondary') is None][0]
    except (KeyError, IndexError):
        pass
    else:
        for option in get_options().values():
            if option.save_in_hobo_as:
                value = options.get(option.name)
                try:
                    resp = requests.post(
                        '/api/variables/',
                        remote_service=hobo_service,
                        data=json.dumps({option.save_in_hobo_as: {'value': value}}),
                        headers={'content-type': 'application/json'},
                        without_user=True,
                        timeout=5,
                    )
                    resp.raise_for_status()
                except RequestException:
                    logger = logging.getLogger('combo.apps.appearance')
                    logger.error('error sending appearance variable to hobo')

    return JsonResponse({'err': 0})
