# combo - content management system
# Copyright (C) 2015-2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.conf import settings
from django.http import JsonResponse
from django.views.generic import TemplateView

from .models import AppearanceSettings


class ThemeOption:
    _value = None
    name = None
    rank = 10

    def __init__(self, **kwargs):
        for k, v in kwargs.items():
            setattr(self, k, v)

        options_values = AppearanceSettings.get_options()
        if options_values:
            if self.type == 'size':
                height_name = self.name.removesuffix('size') + 'height'
                width_name = self.name.removesuffix('size') + 'width'
                if height_name in options_values and width_name in options_values:
                    self.value = {
                        'height': options_values.get(height_name),
                        'width': options_values.get(width_name),
                    }
            elif self.name in options_values:
                self.value = options_values[self.name]

    def __repr__(self):
        return f'<{self.__class__.__name__} {self.name}>'

    @property
    def value(self):
        return self._value or self.default

    @value.setter
    def value(self, value):
        self._value = value


class ManagerHomeView(TemplateView):
    template_name = 'combo/appearance/manager_home.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        available_options = settings.APPEARANCE_OPTIONS or {}

        options = {x: ThemeOption(name=x, **y) for x, y in available_options.items()}
        for option in options.values():
            option_with_default = option
            while getattr(option_with_default, 'default_same_as', None):
                option_with_default = options[option_with_default.default_same_as]
            option.default = option_with_default.default

        context['options'] = list(options.values())
        context['options'].sort(key=lambda x: (x.rank, x.label))
        context['options_prefix'] = settings.APPEARANCE_OPTIONS_PREFIX
        context['iframe_url'] = settings.APPEARANCE_PREVIEW_IFRAME_URL
        return context


def save_options_view(request, *args, **kwargs):
    options = {x: y for x, y in request.POST.items() if x != 'csrfmiddlewaretoken'}

    appearance_settings = AppearanceSettings.singleton()
    appearance_settings.options = options
    appearance_settings.save()

    return JsonResponse({'err': 0})
