# chrono - agendas system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import datetime
import io
import sys
import traceback
import uuid

from django.core.files.base import ContentFile
from django.core.serializers.json import DjangoJSONEncoder
from django.db import connection, models
from django.db.models.fields.json import KT
from django.db.models.functions import Cast, Coalesce, ExtractYear
from django.utils.timezone import now
from django.utils.translation import gettext_lazy as _

from chrono.agendas.models import Agenda, Booking, Event
from chrono.utils.ods import Workbook

STATUS_CHOICES = [
    ('registered', _('Registered')),
    ('running', _('Running')),
    ('failed', _('Failed')),
    ('completed', _('Completed')),
]


def manager_async_job_path(instance, filename):
    return f'manager/result/{instance.uuid}/{filename}'


class ManagerAsyncJob(models.Model):
    uuid = models.UUIDField(default=uuid.uuid4, primary_key=True)

    label = models.CharField(max_length=100)
    status = models.CharField(
        max_length=100,
        default='registered',
        choices=STATUS_CHOICES,
    )
    exception = models.TextField()
    action = models.CharField(max_length=100)
    params = models.JSONField(default=dict, encoder=DjangoJSONEncoder)
    total_count = models.PositiveIntegerField(default=0)
    current_count = models.PositiveIntegerField(default=0)
    result = models.FileField(upload_to=manager_async_job_path, blank=True, null=True)

    creation_timestamp = models.DateTimeField(auto_now_add=True)
    last_update_timestamp = models.DateTimeField(auto_now=True)
    completion_timestamp = models.DateTimeField(default=None, null=True)

    def __str__(self):
        return self.label

    def set_total_count(self, num):
        self.total_count = num
        self.save(update_fields=['total_count'])

    def increment_count(self, amount=1):
        self.current_count = (self.current_count or 0) + amount
        if (now() - self.last_update_timestamp).total_seconds() > 1 or self.current_count >= self.total_count:
            self.save(update_fields=['current_count'])

    def get_completion_status(self):
        current_count = self.current_count or 0

        if not current_count:
            return ''

        if not self.total_count:
            return _('%(current_count)s (unknown total)') % {'current_count': current_count}

        return _('%(current_count)s/%(total_count)s (%(percent)s%%)') % {
            'current_count': int(current_count),
            'total_count': self.total_count,
            'percent': int(current_count * 100 / self.total_count),
        }

    def run(self, spool=False):
        if 'uwsgi' in sys.modules and spool:
            from chrono.utils.spooler import run_manager_async_job

            tenant = getattr(connection, 'tenant', None)
            run_manager_async_job.spool(job_id=str(self.uuid), domain=getattr(tenant, 'domain_url', None))
            return
        elif spool:
            self.refresh_from_db()  # reloads params
        self.status = 'running'
        self.save()
        try:
            getattr(self, self.action)()
        except Exception:  # noqa pylint: disable=broad-except
            self.status = 'failed'
            self.exception = traceback.format_exc()
        finally:
            if self.status == 'running':
                self.status = 'completed'
            if self.status in ['completed', 'failed']:
                self.completion_timestamp = now()
            self.save()

    @property
    def base_template(self):
        if self.action == 'pso_stats':
            return 'chrono/manager_pso_stats.html'

    def pso_stats(self):
        stats = PSOStats(formdata=self.params['formdata'], formsetdata=self.params['formsetdata'])
        ods_file = stats.get_stats(job=self)
        self.result.save(f'{self.params["filename"]}.ods', content=ContentFile(ods_file.getvalue()))
        self.save()


class PSOStats:
    def __init__(self, formdata, formsetdata=None):
        self.formdata = formdata
        self.formsetdata = formsetdata or []
        for key in ['date_start', 'date_end']:
            value = self.formdata[key]
            if isinstance(value, str):
                self.formdata[key] = datetime.datetime.fromisoformat(value).date()

    def get_bookings(self, agenda=None, with_annotation=True):
        data = self.formdata
        date_start = data['date_start']
        date_end = data['date_end']
        booking_qs = Booking.objects.filter(
            event__start_datetime__gte=date_start,
            event__start_datetime__lt=date_end + datetime.timedelta(days=1),
            event__cancelled=False,
            cancellation_datetime__isnull=True,
            user_checks__presence=True,
        ).exclude(user_external_id='')
        if agenda is not None:
            booking_qs = booking_qs.filter(event__agenda=agenda)
        else:
            booking_qs = booking_qs.filter(
                models.Q(event__agenda__in=data['agendas'])
                | models.Q(event__agenda__category__in=data['categories']),
            )
        if with_annotation:
            booking_qs = booking_qs.annotate(
                event_slug=Coalesce('event__primary_event__slug', 'event__slug'),
                duration=models.F('event__duration'),
                age=Cast(
                    ExtractYear(
                        models.Func(
                            Cast('event__start_datetime', output_field=models.DateField()),
                            Cast(
                                KT(f'extra_data__{data["extra_data_dob_key"]}'),
                                output_field=models.DateField(),
                            ),
                            function='AGE',
                        ),
                    ),
                    output_field=models.IntegerField(),
                ),
            )
        return booking_qs

    def get_preview(self):
        plan_key = self.formdata['extra_data_benefit_plan_key']
        family_id_key = self.formdata['extra_data_family_id_key']
        agenda_qs = Agenda.objects.filter(
            models.Q(pk__in=self.formdata['agendas']) | models.Q(category__in=self.formdata['categories']),
        )
        preview = {'concerned_agendas': agenda_qs.count(), 'formset': []}
        all_booking_qs = self.get_bookings()
        for sub_data in self.formsetdata:
            if not sub_data.get('label'):
                continue
            age_op = sub_data['age_operator']
            qs_filters = {}
            plan_value = sub_data['extra_data_benefit_plan_value']
            if plan_value is not None:
                qs_filters[f'extra_data__{plan_key}'] = plan_value
            if age_op != 'all':
                qs_filters[f'age__{age_op}'] = 6
            booking_qs = all_booking_qs.filter(**qs_filters)
            total_booking = booking_qs.aggregate(total=models.Count('id'))
            total_user = (
                booking_qs.values('user_external_id')
                .order_by('user_external_id')
                .distinct()
                .aggregate(total=models.Count('user_external_id'))
            )
            total_family = (
                booking_qs.values(f'extra_data__{family_id_key}')
                .order_by(f'extra_data__{family_id_key}')
                .distinct()
                .count()
            )
            preview['formset'].append(
                {
                    'label': sub_data['label'],
                    'bookings': total_booking['total'],
                    'users': total_user['total'],
                    'families': total_family,
                }
            )
        return preview

    def get_stats(self, job):
        data = self.formdata
        plan_key = data['extra_data_benefit_plan_key']
        family_id_key = data['extra_data_family_id_key']
        date_start = data['date_start']
        date_end = data['date_end']

        writer = Workbook()

        headers = [
            _('Activity'),
            _('Date'),
            _('User category'),
            _('Number of different users'),
            _('Number of different families'),
        ]
        all_event_qs = (
            Event.objects.filter(
                models.Q(agenda__in=data['agendas']) | models.Q(agenda__category__in=data['categories']),
                recurrence_days__isnull=True,
                start_datetime__gte=date_start,
                start_datetime__lt=date_end + datetime.timedelta(days=1),
                cancelled=False,
            )
            .annotate(
                event_slug=Coalesce('primary_event__slug', 'slug'),
            )
            .values('event_slug', 'duration')
            .order_by('event_slug', 'duration')
            .distinct()
        )

        for event in all_event_qs:
            headers.append(_('Nb. of') + ' %s (%s)' % (event['event_slug'], event['duration']))
            headers.append('%s (%s) h' % (event['event_slug'], event['duration']))
        # headers
        writer.writerow(headers, headers=True)

        agenda_qs = Agenda.objects.filter(
            models.Q(pk__in=data['agendas']) | models.Q(category__in=data['categories']),
        )
        job.set_total_count(agenda_qs.count())
        for agenda in agenda_qs:
            all_booking_qs = self.get_bookings(agenda=agenda)
            for sub_data in self.formsetdata:
                if not sub_data.get('label'):
                    continue
                age_op = sub_data['age_operator']
                plan_value = sub_data['extra_data_benefit_plan_value']
                row = {
                    _('Activity'): agenda.label,
                    _('Date'): _('From %(date_start)s to %(date_end)s')
                    % {'date_start': date_start.strftime('%d-%m'), 'date_end': date_end.strftime('%d-%m')},
                    _('User category'): sub_data['label'],
                    _('Number of different users'): 0,
                    _('Number of different families'): 0,
                }
                qs_filters = {}
                if plan_value is not None:
                    qs_filters[f'extra_data__{plan_key}'] = plan_value
                if age_op != 'all':
                    qs_filters[f'age__{age_op}'] = 6

                for event in all_event_qs:
                    event_booking_qs = all_booking_qs.filter(
                        event_slug=event['event_slug'],
                        duration=event['duration'],
                    )
                    if not event_booking_qs.exists():
                        continue
                    booking_qs = event_booking_qs.filter(**qs_filters)
                    total_booking = booking_qs.aggregate(total=models.Count('id'))
                    row[_('Nb. of') + ' %s (%s)' % (event['event_slug'], event['duration'])] = total_booking[
                        'total'
                    ]
                    row['%s (%s) h' % (event['event_slug'], event['duration'])] = round(
                        total_booking['total'] * event['duration'] / 60, 3
                    )
                booking_qs = all_booking_qs.filter(**qs_filters)
                total_user = (
                    booking_qs.values('user_external_id')
                    .order_by('user_external_id')
                    .distinct()
                    .aggregate(total=models.Count('user_external_id'))
                )
                total_family = (
                    booking_qs.values(f'extra_data__{family_id_key}')
                    .order_by(f'extra_data__{family_id_key}')
                    .distinct()
                    .count()
                )
                row[_('Number of different users')] = total_user['total']
                row[_('Number of different families')] = total_family

                writer.writerow([row.get(k) for k in headers])
            job.increment_count()

        ods_file = io.BytesIO()
        writer.save(ods_file)
        return ods_file
