# chrono - agendas system
# Copyright (C) 2016-2024  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.core.exceptions import PermissionDenied
from django.db.models import IntegerField, OuterRef, Subquery
from django.db.models.fields.json import KT
from django.db.models.functions import Cast
from django.views.generic import ListView

from chrono.agendas.models import Event
from chrono.utils.lingo import get_lingo_service

from .forms import JournalFilterSet
from .models import AuditEntry


class JournalHomeView(ListView):
    template_name = 'chrono/journal/home.html'
    paginate_by = 10

    def dispatch(self, request, *args, **kwargs):
        if not request.user.is_staff:
            raise PermissionDenied()
        return super().dispatch(request, *args, **kwargs)

    def get_queryset(self):
        qs = AuditEntry.objects.select_related('agenda', 'user').annotate(
            event_id=Cast(KT('extra_data__event_id'), IntegerField())
        )
        events = Event.objects.filter(
            pk=OuterRef('event_id'),
        ).order_by()
        qs = qs.annotate(
            primary_event_slug=Subquery(events.values('primary_event__slug')[:1]),
            event_slug=Subquery(events.values('slug')[:1]),
            event_date=Subquery(events.values('start_datetime__date')[:1]),
        )
        self.filterset = JournalFilterSet(self.request.GET, queryset=qs)
        return self.filterset.qs

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['filter'] = self.filterset
        context['lingo_enabled'] = bool(get_lingo_service())
        return context


journal_home = JournalHomeView.as_view()
