# chrono - agendas system
# Copyright (C) 2025  Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from django.core.management.base import BaseCommand
from django.db.models import F
from requests import RequestException

from chrono.agendas.models import Agenda


class Command(BaseCommand):
    help = 'Confirm bookings in waiting list'

    def handle(self, **options):
        for agenda in Agenda.objects.filter(kind='events'):
            for event in agenda.event_set.filter(
                agenda__kind='events',
                booked_waiting_list_places__gt=0,
                waiting_list_bookings_automatic_confirm=True,
                booked_places__lt=F('places'),
                start_datetime__gte=agenda.min_booking_datetime,
            ):
                for booking in event.booking_set.filter(
                    in_waiting_list=True,
                    primary_booking__isnull=True,
                    waiting_list_callback_url__isnull=False,
                ).order_by('creation_datetime'):
                    if event.booked_places + booking.places_count() > event.places:
                        # do not allow a more recent booking to be confirmed if the oldest one can not be confirmed due to the event capacity
                        break
                    try:
                        booking.accept(trigger_callback=True)
                    except RequestException:
                        # if a the callback request fails, try the next booking
                        pass
                    else:
                        # we need an up to date value for event.booked_places
                        event.refresh_from_db()
