import pytest

from chrono.agendas.models import Booking
from chrono.utils.timezone import localtime
from tests.utils import build_meetings_agenda

pytestmark = pytest.mark.django_db


@pytest.mark.freeze_time('2025-10-13 06:00')
def test_booking_api_meetings_bypass_delays(app, api_user):
    agenda = build_meetings_agenda(
        'Agenda1',
        meeting_types=(30,),
        desks={
            'desk-1': ['monday-friday 9:00-12:00 14:00-17:00'],
            'desk-2': ['monday-friday 9:00-12:00 14:00-17:00'],
        },
        minimal_booking_delay=2,
        maximal_booking_delay=4,
    )
    resp = app.get(
        '/api/agenda/%s/meetings/%s/datetimes/' % (agenda.slug, agenda.meetingtype_set.first().slug)
    )
    assert len(resp.json['data']) == 24

    # check first and last slots
    assert resp.json['data'][0]['api']['fillslot_url'].endswith('mt-30:2025-10-15-0900/')
    assert resp.json['data'][-1]['api']['fillslot_url'].endswith('mt-30:2025-10-16-1630/')

    # book first and last slot
    Booking.objects.all().delete()
    resp = app.post(
        '/api/agenda/%s/fillslot/mt-30:2025-10-15-0900/' % agenda.slug,
    )
    assert resp.json['err'] == 0
    assert resp.json['datetime'] == '2025-10-15 09:00:00'
    assert (
        localtime(Booking.objects.first().event.start_datetime).strftime('%Y-%m-%d %H:%M:%S')
        == '2025-10-15 09:00:00'
    )

    Booking.objects.all().delete()
    resp = app.post(
        '/api/agenda/%s/fillslot/mt-30:2025-10-16-1630/' % agenda.slug,
    )
    assert resp.json['err'] == 0
    assert resp.json['datetime'] == '2025-10-16 16:30:00'
    assert (
        localtime(Booking.objects.first().event.start_datetime).strftime('%Y-%m-%d %H:%M:%S')
        == '2025-10-16 16:30:00'
    )

    # try to book one day before the first slot and one day after the last slot
    resp = app.post(
        '/api/agenda/%s/fillslot/mt-30:2025-10-14-0900/' % agenda.slug,
    )
    assert resp.json['err'] == 1
    resp = app.post(
        '/api/agenda/%s/fillslot/mt-30:2025-10-17-1630/' % agenda.slug,
    )
    assert resp.json['err'] == 1

    # bypass delays
    Booking.objects.all().delete()
    resp = app.post(
        '/api/agenda/%s/fillslot/mt-30:2025-10-14-0900/' % agenda.slug,
        params={'bypass_delays': True},
    )
    assert resp.json['err'] == 0
    assert resp.json['datetime'] == '2025-10-14 09:00:00'
    assert (
        localtime(Booking.objects.first().event.start_datetime).strftime('%Y-%m-%d %H:%M:%S')
        == '2025-10-14 09:00:00'
    )

    # bypass_delays has no effect on the maximal_booking_delay
    Booking.objects.all().delete()
    resp = app.post(
        '/api/agenda/%s/fillslot/mt-30:2025-10-17-1630/' % agenda.slug,
        params={'bypass_delays': True},
    )
    assert resp.json['err'] == 1
