import datetime

import pytest
from django.utils.timezone import now

from chrono.agendas.models import Agenda, Booking, BookingCheck, Category, Event
from chrono.manager.models import ManagerAsyncJob
from chrono.utils.timezone import make_aware
from tests.utils import get_ods_data_from_file_field, login

pytestmark = pytest.mark.django_db


def test_pso_form(app, admin_user):
    agenda = Agenda.objects.create(label='Foo bar')
    event = Event.objects.create(
        label='event',
        start_datetime=make_aware(datetime.datetime(2022, 2, 1, 12, 0)),
        places=10,
        agenda=agenda,
    )

    login(app)
    resp = app.get('/manage/reports/pso/')
    resp.form['date_start'] = '2022-01-01'
    resp.form['date_end'] = '2021-12-31'
    resp = resp.form.submit()
    assert resp.context['form'].errors['date_end'] == ['End date must be greater than start date.']

    resp.form['date_end'] = '2022-04-02'
    resp = resp.form.submit()
    assert resp.context['form'].errors == {
        'extra_data_dob_key': ['This field is required.'],
        'extra_data_benefit_plan_key': ['This field is required.'],
        'extra_data_family_id_key': ['This field is required.'],
    }

    resp.form['extra_data_dob_key'] = 'dob'
    resp.form['extra_data_benefit_plan_key'] = 'aeeh'
    resp.form['extra_data_family_id_key'] = 'family_num'
    resp.form['date_end'] = '2021-12-31'
    resp = resp.form.submit()
    # no KeyError due to absence of date_end in cleaned_data
    assert resp.context['form'].errors['date_end'] == ['End date must be greater than start date.']

    booking = Booking.objects.create(event=event)
    BookingCheck.objects.create(booking=booking, presence=True)
    resp.form['date_end'] = '2022-04-02'
    resp.form['agendas'] = agenda.slug
    resp = resp.form.submit()
    assert resp.context['form'].errors == {'__all__': ['Can not get statistics: some events has no duration']}

    event.duration = 60
    event.save()
    resp = resp.form.submit()
    assert resp.context['form'].errors == {}

    booking.user_external_id = 'foo'
    booking.save()
    resp = resp.form.submit()
    assert resp.context['form'].errors == {
        '__all__': ['Can not get statistics: extra_data "dob" is missing for 1 bookings']
    }

    booking.extra_data = {'dob': 'foobar'}
    booking.save()
    resp = resp.form.submit()
    assert resp.context['form'].errors == {
        '__all__': ['Can not get statistics: extra_data "dob" has incorrect format value for 1 bookings']
    }

    booking.extra_data = {'dob': '2020-01-01'}
    booking.save()
    resp = resp.form.submit()
    assert resp.context['form'].errors == {
        '__all__': ['Can not get statistics: extra_data "aeeh" is missing for 1 bookings']
    }

    booking.extra_data['aeeh'] = 'foobar'
    booking.save()
    resp = resp.form.submit()
    assert resp.context['form'].errors == {
        '__all__': ['Can not get statistics: extra_data "aeeh" has incorrect format value for 1 bookings']
    }

    booking.extra_data['aeeh'] = True
    booking.save()
    resp = resp.form.submit()
    assert resp.context['form'].errors == {
        '__all__': ['Can not get statistics: extra_data "family_num" is missing for 1 bookings']
    }

    booking.extra_data['family_num'] = ''
    booking.save()
    resp = resp.form.submit()
    assert resp.context['form'].errors == {
        '__all__': [
            'Can not get statistics: extra_data "family_num" has incorrect format value for 1 bookings'
        ]
    }

    booking.extra_data['family_num'] = '42'
    booking.save()
    resp = resp.form.submit()
    assert resp.context['form'].errors == {}
    assert resp.context['preview'] == {'concerned_agendas': 1, 'formset': []}


def test_pso_formset(app, admin_user):
    agenda = Agenda.objects.create(label='Foo bar')
    event = Event.objects.create(
        label='event',
        start_datetime=make_aware(datetime.datetime(2022, 2, 1, 12, 0)),
        places=10,
        duration=60,
        agenda=agenda,
    )
    booking = Booking.objects.create(
        event=event,
        user_external_id='a',
        extra_data={'dob': '2015-02-01', 'aeeh': False, 'family_num': '1'},  # 7 years
    )
    BookingCheck.objects.create(booking=booking, presence=True)
    booking = Booking.objects.create(
        event=event,
        user_external_id='b',
        extra_data={'dob': '2015-02-01', 'aeeh': False, 'family_num': '1'},  # 7 years
    )
    BookingCheck.objects.create(booking=booking, presence=True)
    booking = Booking.objects.create(
        event=event,
        user_external_id='c',
        extra_data={'dob': '2016-02-01', 'aeeh': False, 'family_num': '2'},  # 6 years
    )
    BookingCheck.objects.create(booking=booking, presence=True)
    booking = Booking.objects.create(
        event=event,
        user_external_id='a',
        extra_data={'dob': '2016-02-02', 'aeeh': False, 'family_num': '1'},  # 5 years
    )
    BookingCheck.objects.create(booking=booking, presence=True)
    booking = Booking.objects.create(
        event=event,
        user_external_id='a',
        extra_data={'dob': '2015-02-01', 'aeeh': True, 'family_num': '1'},  # 7 years
    )
    BookingCheck.objects.create(booking=booking, presence=True)
    booking = Booking.objects.create(
        event=event,
        user_external_id='a',
        extra_data={'dob': '2016-02-01', 'aeeh': True, 'family_num': '1'},  # 6 years
    )
    BookingCheck.objects.create(booking=booking, presence=True)
    booking = Booking.objects.create(
        event=event,
        user_external_id='a',
        extra_data={'dob': '2016-02-02', 'aeeh': True, 'family_num': '1'},  # 5 years
    )
    BookingCheck.objects.create(booking=booking, presence=True)

    query_params = {
        'date_start': '2022-01-01',
        'date_end': '2022-04-02',
        'agendas': agenda.slug,
        'extra_data_dob_key': 'dob',
        'extra_data_benefit_plan_key': 'aeeh',
        'extra_data_family_id_key': 'family_num',
        'form-TOTAL_FORMS': 7,
        'form-INITIAL_FORMS': 0,
        'form-MIN_NUM_FORMS': 0,
        'form-MAX_NUM_FORMS': 1000,
        'form-0-label': '< 6',
        'form-0-age_operator': 'lt',
        'form-0-extra_data_benefit_plan_value': False,
        'form-1-label': '<= 6',
        'form-1-age_operator': 'lte',
        'form-1-extra_data_benefit_plan_value': False,
        'form-2-label': '> 6',
        'form-2-age_operator': 'gt',
        'form-2-extra_data_benefit_plan_value': False,
        'form-3-label': '>= 6',
        'form-3-age_operator': 'gte',
        'form-3-extra_data_benefit_plan_value': False,
        'form-4-label': 'aeeh',
        'form-4-age_operator': 'all',
        'form-4-extra_data_benefit_plan_value': True,
        'form-5-label': '< 6 all',
        'form-5-age_operator': 'lt',
        'form-5-extra_data_benefit_plan_value': '',
        'form-6-label': '>= 6 all',
        'form-6-age_operator': 'gte',
        'form-6-extra_data_benefit_plan_value': '',
    }
    login(app)
    resp = app.get('/manage/reports/pso/', params=query_params)
    assert resp.context['form'].errors == {}
    assert resp.context['preview'] == {
        'concerned_agendas': 1,
        'formset': [
            {'label': '< 6', 'bookings': 1, 'users': 1, 'families': 1},
            {'label': '<= 6', 'bookings': 2, 'users': 2, 'families': 2},
            {'label': '> 6', 'bookings': 2, 'users': 2, 'families': 1},
            {'label': '>= 6', 'bookings': 3, 'users': 3, 'families': 2},
            {'label': 'aeeh', 'bookings': 3, 'users': 1, 'families': 1},
            {'label': '< 6 all', 'bookings': 2, 'users': 1, 'families': 1},
            {'label': '>= 6 all', 'bookings': 5, 'users': 3, 'families': 2},
        ],
    }


def test_pso_ods(app, admin_user):
    category = Category.objects.create(label='cat')
    agenda1 = Agenda.objects.create(label='Foo bar', category=category)
    agenda2 = Agenda.objects.create(label='Foo bar2')
    event1 = Event.objects.create(
        label='event1',
        start_datetime=make_aware(datetime.datetime(2022, 2, 1, 12, 0)),
        duration=60,
        places=10,
        agenda=agenda1,
    )
    event2 = Event.objects.create(
        label='event2',
        start_datetime=make_aware(datetime.datetime(2022, 2, 1, 16, 0)),
        duration=120,
        places=10,
        agenda=agenda1,
    )
    event3 = Event.objects.create(
        label='event3',
        start_datetime=make_aware(datetime.datetime(2022, 2, 1, 16, 0)),
        duration=30,
        places=10,
        agenda=agenda2,
    )
    cancelled_event = Event.objects.create(
        label='cancelled',
        start_datetime=make_aware(datetime.datetime(2022, 2, 1, 16, 0)),
        duration=30,
        places=10,
        agenda=agenda2,
        cancelled=True,
    )
    # cancelled event, ignored
    booking = Booking.objects.create(
        event=cancelled_event,
        user_external_id='a',
        extra_data={'dob': '2015-02-01', 'aeeh': False, 'family_num': '1'},  # 7 years
    )
    BookingCheck.objects.create(booking=booking, presence=True)
    recurring_event = Event.objects.create(
        label='recurring',
        start_datetime=make_aware(datetime.datetime(2022, 2, 1, 8, 0)),
        duration=60,
        places=10,
        agenda=agenda2,
        recurrence_days=[1, 2, 3, 4, 5, 6, 7],
        recurrence_end_date=datetime.date(2022, 2, 3),
    )
    recurring_event.create_all_recurrences()
    recurrences = recurring_event.recurrences.all()
    recurrent_event1 = recurrences[0]
    recurrent_event2 = recurrences[1]

    for event in [event1, event2, recurrent_event1, recurrent_event2]:
        booking = Booking.objects.create(
            event=event,
            user_external_id='a',
            extra_data={'dob': '2015-02-01', 'aeeh': False, 'family_num': '1'},  # 7 years
        )
        BookingCheck.objects.create(booking=booking, presence=True)
        # no check, ignored
        Booking.objects.create(
            event=event,
            user_external_id='a',
            extra_data={'dob': '2015-02-01', 'aeeh': False, 'family_num': '1'},
        )
        # absence, ignored
        booking = Booking.objects.create(
            event=event,
            user_external_id='a',
            extra_data={'dob': '2015-02-01', 'aeeh': False, 'family_num': '1'},
        )
        BookingCheck.objects.create(booking=booking, presence=False)
        # cancelled, ignored
        booking = Booking.objects.create(
            event=event,
            user_external_id='a',
            cancellation_datetime=now(),
            extra_data={'dob': '2015-02-01', 'aeeh': False},
        )
        BookingCheck.objects.create(booking=booking, presence=True)
    for event in [event3, recurrent_event1, recurrent_event2]:
        booking = Booking.objects.create(
            event=event,
            user_external_id='b',
            extra_data={'dob': '2015-09-01', 'aeeh': False, 'family_num': '1'},  # 6 years
        )
        BookingCheck.objects.create(booking=booking, presence=True)
    for event in [event3, recurrent_event1, recurrent_event2]:
        booking = Booking.objects.create(
            event=event,
            user_external_id='c',
            extra_data={'dob': '2016-02-01', 'aeeh': False, 'family_num': '2'},  # 6 years
        )
        BookingCheck.objects.create(booking=booking, presence=True)
    for event in [event1, recurrent_event1, recurrent_event2]:
        booking = Booking.objects.create(
            event=event,
            user_external_id='d',
            extra_data={'dob': '2016-02-02', 'aeeh': False, 'family_num': '3'},  # 5 or 6 years
        )
        BookingCheck.objects.create(booking=booking, presence=True)
    for event in [event1, event3, recurrent_event2]:
        booking = Booking.objects.create(
            event=event,
            user_external_id='e',
            extra_data={'dob': '2015-02-01', 'aeeh': True, 'family_num': '3'},
        )
        BookingCheck.objects.create(booking=booking, presence=True)
    for event in [event1, event3, recurrent_event2]:
        booking = Booking.objects.create(
            event=event,
            user_external_id='f',
            extra_data={'dob': '2016-09-01', 'aeeh': True, 'family_num': '4'},
        )
        BookingCheck.objects.create(booking=booking, presence=True)
    for event in [event1, event2, event3, recurrent_event1]:
        booking = Booking.objects.create(
            event=event,
            user_external_id='g',
            extra_data={'dob': '2016-02-15', 'aeeh': True, 'family_num': '5'},
        )
        BookingCheck.objects.create(booking=booking, presence=True)

    query_params = {
        'date_start': '2022-02-01',
        'date_end': '2022-02-28',
        'agendas': agenda2.slug,
        'categories': category.slug,
        'extra_data_dob_key': 'dob',
        'extra_data_benefit_plan_key': 'aeeh',
        'extra_data_family_id_key': 'family_num',
        'form-TOTAL_FORMS': 7,
        'form-INITIAL_FORMS': 0,
        'form-MIN_NUM_FORMS': 0,
        'form-MAX_NUM_FORMS': 1000,
        'form-0-label': '< 6',
        'form-0-age_operator': 'lt',
        'form-0-extra_data_benefit_plan_value': False,
        'form-1-label': '<= 6',
        'form-1-age_operator': 'lte',
        'form-1-extra_data_benefit_plan_value': False,
        'form-2-label': '> 6',
        'form-2-age_operator': 'gt',
        'form-2-extra_data_benefit_plan_value': False,
        'form-3-label': '>= 6',
        'form-3-age_operator': 'gte',
        'form-3-extra_data_benefit_plan_value': False,
        'form-4-label': 'aeeh',
        'form-4-age_operator': 'all',
        'form-4-extra_data_benefit_plan_value': True,
        'form-5-label': '< 6 all',
        'form-5-age_operator': 'lt',
        'form-5-extra_data_benefit_plan_value': '',
        'form-6-label': '>= 6 all',
        'form-6-age_operator': 'gte',
        'form-6-extra_data_benefit_plan_value': '',
    }
    login(app)
    resp = app.get('/manage/reports/pso/', params=query_params)
    assert resp.context['form'].errors == {}
    assert resp.context['preview'] == {
        'concerned_agendas': 2,
        'formset': [
            {'label': '< 6', 'bookings': 2, 'users': 1, 'families': 1},
            {'label': '<= 6', 'bookings': 9, 'users': 3, 'families': 3},
            {'label': '> 6', 'bookings': 4, 'users': 1, 'families': 1},
            {'label': '>= 6', 'bookings': 11, 'users': 4, 'families': 3},
            {'label': 'aeeh', 'bookings': 10, 'users': 3, 'families': 3},
            {'label': '< 6 all', 'bookings': 9, 'users': 3, 'families': 3},
            {'label': '>= 6 all', 'bookings': 14, 'users': 5, 'families': 3},
        ],
    }

    query_params['ods'] = ''
    resp = app.get('/manage/reports/pso/', params=query_params)
    job = ManagerAsyncJob.objects.latest('creation_timestamp')
    assert job.status == 'completed'
    assert job.get_completion_status() == '2/2 (100%)'
    assert resp.location.endswith('/manage/job/%s/' % job.uuid)
    resp = resp.follow()
    assert job.result.url in resp
    sheets = get_ods_data_from_file_field(job.result)
    assert len(sheets) == 1
    rows = list(sheets.values())[0]
    assert len(rows) == 15
    assert rows == [
        [
            'Activity',
            'Date',
            'User category',
            'Number of different users',
            'Number of different families',
            'Nb. of event1 (60)',
            'event1 (60) h',
            'Nb. of event2 (120)',
            'event2 (120) h',
            'Nb. of event3 (30)',
            'event3 (30) h',
            'Nb. of recurring (60)',
            'recurring (60) h',
        ],
        ['Foo bar', 'From 01-02 to 28-02', '< 6', '1', '1', '1', '1.0', '0', '0.0'],
        ['Foo bar', 'From 01-02 to 28-02', '<= 6', '1', '1', '1', '1.0', '0', '0.0'],
        ['Foo bar', 'From 01-02 to 28-02', '> 6', '1', '1', '1', '1.0', '1', '2.0'],
        ['Foo bar', 'From 01-02 to 28-02', '>= 6', '1', '1', '1', '1.0', '1', '2.0'],
        ['Foo bar', 'From 01-02 to 28-02', 'aeeh', '3', '3', '3', '3.0', '1', '2.0'],
        ['Foo bar', 'From 01-02 to 28-02', '< 6 all', '3', '3', '3', '3.0', '1', '2.0'],
        ['Foo bar', 'From 01-02 to 28-02', '>= 6 all', '2', '2', '2', '2.0', '1', '2.0'],
        ['Foo bar2', 'From 01-02 to 28-02', '< 6', '1', '1', '', '', '', '', '0', '0.0', '1', '1.0'],
        ['Foo bar2', 'From 01-02 to 28-02', '<= 6', '3', '3', '', '', '', '', '2', '1.0', '6', '6.0'],
        ['Foo bar2', 'From 01-02 to 28-02', '> 6', '1', '1', '', '', '', '', '0', '0.0', '2', '2.0'],
        ['Foo bar2', 'From 01-02 to 28-02', '>= 6', '4', '3', '', '', '', '', '2', '1.0', '7', '7.0'],
        ['Foo bar2', 'From 01-02 to 28-02', 'aeeh', '3', '3', '', '', '', '', '3', '1.5', '3', '3.0'],
        ['Foo bar2', 'From 01-02 to 28-02', '< 6 all', '3', '3', '', '', '', '', '2', '1.0', '3', '3.0'],
        ['Foo bar2', 'From 01-02 to 28-02', '>= 6 all', '5', '3', '', '', '', '', '3', '1.5', '8', '8.0'],
    ]

    query_params['date_end'] = '2022-02-01'
    resp = app.get('/manage/reports/pso/', params=query_params)
    job = ManagerAsyncJob.objects.latest('creation_timestamp')
    assert job.status == 'completed'
    assert job.get_completion_status() == '2/2 (100%)'
    assert resp.location.endswith('/manage/job/%s/' % job.uuid)
    resp = resp.follow()
    assert job.result.url in resp
    sheets = get_ods_data_from_file_field(job.result)
    assert len(sheets) == 1
    rows = list(sheets.values())[0]
    assert len(rows) == 15
    assert rows == [
        [
            'Activity',
            'Date',
            'User category',
            'Number of different users',
            'Number of different families',
            'Nb. of event1 (60)',
            'event1 (60) h',
            'Nb. of event2 (120)',
            'event2 (120) h',
            'Nb. of event3 (30)',
            'event3 (30) h',
            'Nb. of recurring (60)',
            'recurring (60) h',
        ],
        ['Foo bar', 'From 01-02 to 01-02', '< 6', '1', '1', '1', '1.0', '0', '0.0'],
        ['Foo bar', 'From 01-02 to 01-02', '<= 6', '1', '1', '1', '1.0', '0', '0.0'],
        ['Foo bar', 'From 01-02 to 01-02', '> 6', '1', '1', '1', '1.0', '1', '2.0'],
        ['Foo bar', 'From 01-02 to 01-02', '>= 6', '1', '1', '1', '1.0', '1', '2.0'],
        ['Foo bar', 'From 01-02 to 01-02', 'aeeh', '3', '3', '3', '3.0', '1', '2.0'],
        ['Foo bar', 'From 01-02 to 01-02', '< 6 all', '3', '3', '3', '3.0', '1', '2.0'],
        ['Foo bar', 'From 01-02 to 01-02', '>= 6 all', '2', '2', '2', '2.0', '1', '2.0'],
        ['Foo bar2', 'From 01-02 to 01-02', '< 6', '1', '1', '', '', '', '', '0', '0.0', '1', '1.0'],
        ['Foo bar2', 'From 01-02 to 01-02', '<= 6', '3', '3', '', '', '', '', '2', '1.0', '3', '3.0'],
        ['Foo bar2', 'From 01-02 to 01-02', '> 6', '1', '1', '', '', '', '', '0', '0.0', '1', '1.0'],
        ['Foo bar2', 'From 01-02 to 01-02', '>= 6', '3', '2', '', '', '', '', '2', '1.0', '3', '3.0'],
        ['Foo bar2', 'From 01-02 to 01-02', 'aeeh', '3', '3', '', '', '', '', '3', '1.5', '1', '1.0'],
        ['Foo bar2', 'From 01-02 to 01-02', '< 6 all', '3', '3', '', '', '', '', '2', '1.0', '2', '2.0'],
        ['Foo bar2', 'From 01-02 to 01-02', '>= 6 all', '4', '3', '', '', '', '', '3', '1.5', '3', '3.0'],
    ]


def test_pso_as_manager(app, manager_user):
    agenda = Agenda.objects.create(label='Foo bar')
    app = login(app, username='manager', password='manager')
    resp = app.get('/manage/reports/pso/')
    assert resp.form['agendas'].options == [('', True, '---------')]
    assert resp.form['categories'].options == [('', True, '---------')]

    agenda.view_role = manager_user.groups.all()[0]
    agenda.save()
    resp = app.get('/manage/reports/pso/')
    assert resp.form['agendas'].options == [('', True, '---------'), ('foo-bar', False, 'Foo bar')]
    assert resp.form['categories'].options == [('', True, '---------')]

    category = Category.objects.create(label='cat')
    agenda.category = category
    agenda.save()
    resp = app.get('/manage/reports/pso/')
    assert resp.form['agendas'].options == [('', True, '---------'), ('foo-bar', False, '(cat) Foo bar')]
    assert resp.form['categories'].options == [('', True, '---------'), ('cat', False, 'cat')]
