from django.db import models
from django.db.models import Case, F, When
from django.db.models.functions import Cast, Coalesce
from django.utils.translation import gettext_lazy as _


def get_group_criterias():
    return GroupByCriteria.__subclasses__()


class GroupByCriteria:
    @staticmethod
    def annotate_bookings(qs):
        return qs

    @staticmethod
    def get_label(group_value):
        return str(group_value) or _('None')


class AgendaCriteria(GroupByCriteria):
    label = _('Agenda')
    slug = 'agenda'
    lookup = 'event__agenda__label'


class EventCriteria(GroupByCriteria):
    label = _('Event')
    slug = 'event'
    lookup = 'event__label'


class PresenceCriteria(GroupByCriteria):
    label = _('Presence/Absence')
    slug = 'user_was_present'
    lookup = 'presence'
    labels = {
        None: _('Booked'),
        True: _('Present'),
        False: _('Absent'),
    }

    @staticmethod
    def annotate_bookings(qs):
        return qs.annotate(
            presence=Case(
                When(primary_booking__isnull=True, then=F('user_checks__presence')),
                When(primary_booking__isnull=False, then=F('primary_booking__user_checks__presence')),
            )
        )

    @classmethod
    def get_label(cls, group_value):
        return cls.labels[group_value]


class CheckTypeCriteria(GroupByCriteria):
    label = _('Check type')
    slug = 'check_type'
    lookup = 'check_type'
    labels = {
        None: _('Not checked'),
        'true': _('Present (no check type)'),
        'false': _('Absent (no check type)'),
    }

    @staticmethod
    def annotate_bookings(qs):
        qs = PresenceCriteria.annotate_bookings(qs)
        return qs.annotate(
            check_type_label=Case(
                When(primary_booking__isnull=True, then=F('user_checks__type_label')),
                When(primary_booking__isnull=False, then=F('primary_booking__user_checks__type_label')),
            ),
            check_type=Coalesce('check_type_label', Cast('presence', models.CharField())),
        )

    @classmethod
    def get_label(cls, group_value):
        return cls.labels.get(group_value, group_value)
