from unittest import mock

import django_webtest
import pytest
from django.contrib.auth import get_user_model
from django.core.signals import setting_changed
from django.dispatch import receiver

from chrono.utils.timezone import get_default_timezone

from .utils import get_ods_data

User = get_user_model()


@pytest.fixture
def user():
    user, _ = User.objects.get_or_create(
        username='john.doe', first_name='John', last_name='Doe', email='john.doe@example.net'
    )
    user.set_password('password')
    user.save()
    return user


class PublikTestResponse(django_webtest.DjangoWebtestResponse):

    @property
    def ods_data(self):
        assert self.content_type == 'application/vnd.oasis.opendocument.spreadsheet', self.headers
        assert self.headers['Content-Disposition'].startswith('attachment; filename="')
        assert self.headers['Content-Disposition'].endswith('.ods"')
        return get_ods_data(self)


django_webtest.DjangoTestApp.response_class = PublikTestResponse


@pytest.fixture
def app(request, user):
    wtm = django_webtest.WebTestMixin()
    wtm.setup_auth = False
    wtm._patch_settings()
    request.addfinalizer(wtm._unpatch_settings)
    app = django_webtest.DjangoTestApp()
    app.authorization = ('Basic', ('john.doe', 'password'))
    return app


@pytest.fixture(autouse=True)
def media_root(settings, tmpdir):
    settings.MEDIA_ROOT = str(tmpdir.mkdir('media_root'))


@pytest.fixture
def nocache(settings):
    settings.CACHES = {
        'default': {
            'BACKEND': 'django.core.cache.backends.dummy.DummyCache',
        }
    }


@receiver(setting_changed)
def update_connections_time_zone(**kwargs):
    if kwargs['setting'] == 'TIME_ZONE':
        # Reset local time zone lru cache
        get_default_timezone.cache_clear()


@pytest.fixture
def admin_user(db):
    try:
        user = User.objects.get(username='admin')
    except User.DoesNotExist:
        user = User.objects.create_superuser('admin', email=None, password='admin')
    return user


@pytest.fixture
def api_user(db):
    try:
        return User.objects.get(username='api-user')
    except User.DoesNotExist:
        apiclient = User.objects.create(
            username='api-user', first_name='API', last_name='User', email='apiclient@example.net'
        )
        apiclient.set_password('password')
        apiclient.save()

    def fake_is_apiclient(user):
        if user and user.id == apiclient.id:
            return True
        return False

    with mock.patch('chrono.apps.journal.utils.is_apiclient', wraps=fake_is_apiclient):
        yield apiclient
