# authentic2 - versatile identity manager
# Copyright (C) 2010-2022 Entr'ouvert
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import pytest

from authentic2_auth_fc.models import FcAuthenticator
from tests.utils import login


@pytest.mark.parametrize('full_scopes_display', [True, False])
@pytest.mark.parametrize('email_linking_option', [True, False])
@pytest.mark.parametrize('authn_links_by_email', [True, False])
def test_authenticators_fc(
    app, superuser, settings, fc2_jwkset_mock, full_scopes_display, email_linking_option, authn_links_by_email
):
    if full_scopes_display:
        settings.A2_FC_DISPLAY_COMMON_SCOPES_ONLY = False
    if email_linking_option:
        settings.A2_FC_DISPLAY_EMAIL_LINKING_OPTION = True
    resp = login(app, superuser, path='/manage/authenticators/')

    resp = resp.click('Add new authenticator')
    resp.form['authenticator'] = 'fc'
    resp = resp.form.submit()
    assert '/edit/' in resp.location

    provider = FcAuthenticator.objects.get()
    provider.link_by_email = authn_links_by_email
    provider.platform = 'test'
    provider.save(update_fields=['link_by_email', 'platform'])
    assert provider.order == -1
    assert not provider.supports_multiaccount

    resp = app.get(provider.get_absolute_url())
    assert 'extra-actions-menu-opener' in resp.text
    assert 'Platform: Integration' in resp.text
    assert 'Scopes: profile (profile), email (email)' in resp.text
    assert 'Client Identifier' not in resp.text
    assert 'Client Secret' not in resp.text

    resp = resp.click('Edit')
    if email_linking_option or authn_links_by_email:
        assert list(resp.form.fields) == [
            'csrfmiddlewaretoken',
            'show_condition',
            'platform',
            'version',
            'client_id',
            'client_secret',
            'scopes',
            'link_by_email',
            'supports_multiaccount',
            None,
        ]
    else:
        assert list(resp.form.fields) == [
            'csrfmiddlewaretoken',
            'show_condition',
            'platform',
            'version',
            'client_id',
            'client_secret',
            'scopes',
            'supports_multiaccount',
            None,
        ]
    assert 'phone' not in resp.pyquery('#id_scopes').html()
    assert 'address' not in resp.pyquery('#id_scopes').html()
    assert {option[0] for option in resp.form.fields['platform'][0].options} == {'test', 'prod', 'tnew'}

    # django 3 and 4 rendered html discrepancies
    scopes_list = resp.pyquery('#id_scopes li') or resp.pyquery('#id_scopes div')

    if full_scopes_display:
        assert {scope.text_content().strip() for scope in scopes_list} == {
            'given name (given_name)',
            'gender (gender)',
            'birthdate (birthdate)',
            'birthcountry (birthcountry)',
            'birthplace (birthplace)',
            'family name (family_name)',
            'email (email)',
            'usual family name (preferred_username)',
            'core identity (identite_pivot)',
            'profile (profile)',
            'birth profile (birth)',
            'given name (from the RNIPP)',
            'family name (from the RNIPP)',
            'gender (from the RNIPP)',
            'birthcountry (from the RNIPP)',
            'birthplace (from the RNIPP)',
            'birthdate (from the RNIPP)',
            'profile (from the RNIPP)',
            'core identity (from the RNIPP)',
        }
    else:
        assert {scope.text_content().strip() for scope in scopes_list} == {
            'family name (family_name)',
            'given name (given_name)',
            'birthdate (birthdate)',
            'birthplace (birthplace)',
            'birthcountry (birthcountry)',
            'profile (profile)',
            'gender (gender)',
            'usual family name (preferred_username)',
            'email (email)',
        }

    resp.form['platform'] = 'prod'
    resp.form['client_id'] = '211286433e39cce01db448d80181bdfd005554b19cd51b3fe7943f6b3b86ab6k'
    resp.form['client_secret'] = '211286433e39cce01db448d80181bdfd005554b19cd51b3fe7943f6b3b86ab6d'
    resp.form['scopes'] = ['given_name', 'birthdate']
    resp.form['supports_multiaccount'] = True
    resp = resp.form.submit().follow()

    provider.refresh_from_db()
    assert provider.supports_multiaccount

    assert 'Platform: Production' in resp.text
    assert 'Scopes: given name (given_name), birthdate (birthdate)' in resp.text
    assert 'Client Identifier: 211286433e39cce01db448d80181bdfd005554b19cd51b3fe7943f6b3b86ab6k' in resp.text
    assert 'Client Secret: 211286433e39cce01db448d80181bdfd005554b19cd51b3fe7943f6b3b86ab6d' in resp.text

    resp = app.get('/manage/authenticators/')
    assert 'FranceConnect' in resp.text
    assert 'class="section disabled"' in resp.text

    resp = resp.click('Configure', index=1)
    resp = resp.click('Enable').follow()
    assert 'Authenticator has been enabled.' in resp.text

    resp = app.get('/manage/authenticators/')
    assert 'class="section disabled"' not in resp.text

    provider.refresh_from_db()
    provider.scopes.extend(['phone', 'address'])  # deprecated scopes
    provider.save()

    resp = app.get(provider.get_absolute_url())
    resp = resp.click('Edit')
    resp.form.submit().follow()
    assert {option[0] for option in resp.form.fields['platform'][0].options} == {'prod', 'tnew'}
    provider.refresh_from_db()
    assert 'phone' not in provider.scopes
    assert 'address' not in provider.scopes

    if not full_scopes_display:
        provider.refresh_from_db()
        # 2 valid scopes and 2 invalid ones
        provider.scopes.extend(['profile', 'rnipp_identite_pivot', 'address', 'baz'])
        provider.save()
        resp = app.get(provider.get_absolute_url())
        resp = resp.click('Edit')
        scopes_list = resp.pyquery('#id_scopes li') or resp.pyquery('#id_scopes div')
        # only the valid scopes are added to the form
        assert {scope.text_content().strip() for scope in scopes_list} == {
            'given name (given_name)',
            'family name (family_name)',
            'birthcountry (birthcountry)',
            'birthplace (birthplace)',
            'gender (gender)',
            'usual family name (preferred_username)',
            'email (email)',
            'birthdate (birthdate)',
            'profile (profile)',
            'core identity (from the RNIPP)',
        }
