# ANTS-Hub - Copyright (C) Entr'ouvert

import logging

try:
    import uwsgi
except ImportError:
    uwsgi = None

# inspired by https://github.com/unbit/uwsgi/blob/master/uwsgidecorators.py

logger = logging.getLogger('ants_hub')


def get_free_signal():
    if not uwsgi:
        return -1

    for signum in range(0, 256):
        if not uwsgi.signal_registered(signum):
            return signum
    raise RuntimeError('No free uwsgi signal available')


def log_exception(func, name):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception:
            log_msg = f'Exception during {name}'
            logger.exception(log_msg)

    return wrapper


class cron:
    def __init__(
        self, minute, hour, day, month, dayweek, **kwargs
    ):  # pylint: disable=too-many-positional-arguments
        self.num = kwargs.get('signum', get_free_signal())
        self.minute = minute
        self.hour = hour
        self.day = day
        self.month = month
        self.dayweek = dayweek
        self.target = kwargs.get('target', '')

    def __call__(self, f):
        if not uwsgi:
            return f

        uwsgi.register_signal(self.num, self.target, log_exception(f, f'CRON {f}'))
        uwsgi.add_cron(self.num, self.minute, self.hour, self.day, self.month, self.dayweek)
        return f


@cron(-1, -1, -1, -1, -1, target='workers')
def check_if_ants_doublon_is_down(num):
    from ants_hub.api.ants import ants_doublon_is_down

    uwsgi.log('CRON ants doublon is down...')
    ants_doublon_is_down()
